/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2018-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.ds.config.RecurringTaskConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ReadOnlyEntry;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions that may be
 * used to schedule administrative tasks on a recurring basis.  Recurring
 * tasks are part of a recurring task chain, which combines the configuration
 * needed to define the schedule with the recurring tasks that define the
 * content of the tasks to schedule.
 * <BR><BR>
 * Recurring tasks created via this API can only be used to schedule
 * {@link Task} instances that have themselves been implemented using the Server
 * SDK.  As when scheduling an individual instance of these tasks, the server
 * must be configured as follows to allow creation of third-party tasks:
 * <PRE>
 *   dsconfig set-global-configuration-prop
 *     --add allowed-task:com.unboundid.directory.sdk.extensions.ThirdPartyTask
 * </PRE>
 * <BR><BR>
 * The {@code Task} instance actually performs the core processing for the
 * administrative task, while the {@code RecurringTask} instance merely ensures
 * that the {@code Task} is scheduled with the appropriate arguments.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class RecurringTask
       implements UnboundIDExtension, Reconfigurable<RecurringTaskConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this recurring task.  All recurring task
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@link #initializeRecurringTask} method.
   */
  public RecurringTask()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  @Override()
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this recurring task.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this recurring task.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this recurring task.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         recurring task.
   */
  public void initializeRecurringTask(
                   final DirectoryServerContext serverContext,
                   final RecurringTaskConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public boolean isConfigurationAcceptable(final RecurringTaskConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public ResultCode applyConfiguration(final RecurringTaskConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this recurring task is
   * to be taken out of service.
   */
  public void finalizeRecurringTask()
  {
    // No implementation is required.
  }



  /**
   * Retrieves the Java class that will be used to process instances of this
   * recurring task.
   *
   * @return  The java class that will be used to process instances of this
   *          recurring task.  It must not be {@code null}.
   */
  public abstract Class<? extends Task> getTaskJavaClass();



  /**
   * Retrieves the list of values that should be provided to the
   * ds-third-party-task-argument attribute for the task instance that is
   * created.
   *
   * @param  scheduledStartTime  The scheduled start time that will be used for
   *                             the task that is created.
   * @param  lastTaskEntry       The entry for the last instance of the
   *                             recurring task that was scheduled.  It may be
   *                             {@code null} if the last instance is not
   *                             available (for example, because no instances of
   *                             the recurring task have yet been scheduled.
   *
   * @return  The list of values that should be provided to the
   *          ds-third-party-task-argument for the task instance that is
   *          created.  It may be {@code null} or empty if the task does not
   *          require any arguments.
   */
  public abstract List<String> getTaskArguments(
                                    final ZonedDateTime scheduledStartTime,
                                    final ReadOnlyEntry lastTaskEntry);



  /**
   * {@inheritDoc}
   */
  @Override()
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
