/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2012-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.SASLBindRequest;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.ds.config.SASLMechanismHandlerConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.ds.types.SASLBindResult;
import com.unboundid.directory.sdk.ds.types.SASLBindResultFactory;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which add
 * support for custom SASL mechanisms to the server, which can provide support
 * for custom authentication (and optionally, authorization) methods.
 * <BR>
 * <H2>Configuring SASL Mechanism Handlers</H2>
 * In order to configure a SASL mechanism handler created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-sasl-mechanism-handler \
 *           --handler-name "<I>{handler-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{handler-name}</I>" is the name to use for the SASL mechanism
 * handler instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.ds.api.SASLMechanismHandler}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the handler.  If multiple arguments should be provided to the SASL
 * mechanism handler, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true,
     notes="Any SASL bind requests received by the Directory Proxy Server " +
          "will be processed by the Directory Proxy Server itself rather " +
          "than being forwarded to backend servers.  However, a SASL " +
          "mechanism handler running in the Directory Proxy Server may " +
          "perform internal operations which reference content in the " +
          "backend servers as if it was contained locally in the server.")
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class SASLMechanismHandler
       implements UnboundIDExtension,
                  Reconfigurable<SASLMechanismHandlerConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this SASL mechanism handler.  All SASL
   * mechanism handler implementations must include a default constructor, but
   * any initialization
   * should generally be done in the {@code initializeSASLMechanismHandler}
   * method.
   */
  public SASLMechanismHandler()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this SASL mechanism handler.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this SASL mechanism
   *                        handler.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this SASL mechanism handler.
   *
   * @throws  LDAPException  If a problem occurs while initializing this SASL
   *                         mechanism handler.
   */
  public void initializeSASLMechanismHandler(
                   final DirectoryServerContext serverContext,
                   final SASLMechanismHandlerConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final SASLMechanismHandlerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final SASLMechanismHandlerConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this SASL " +
                "mechanism handler is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this SASL mechanism
   * handler is to be taken out of service.
   */
  public void finalizeSASLMechanismHandler()
  {
    // No implementation is required.
  }



  /**
   * Retrieves a list of the names of the SASL mechanisms supported by this
   * SASL mechanism handler.  This method will be invoked only immediately after
   * the {@link #initializeSASLMechanismHandler} method is called.
   *
   * @return  A list of the names of the SASL mechanisms supported by this SASL
   *          mechanism handler.
   */
  public abstract List<String> getSASLMechanismNames();



  /**
   * Indicates whether the SASL authentication process using the specified
   * mechanism may be considered secure (i.e., that a third party able to
   * observe the communication, potentially over an insecure communication
   * channel, would not be able to reproduce the authentication process).
   *
   * @param  mechanism  The name of the mechanism for which to make the
   *                    determination.  This will only be invoked with names of
   *                    mechanisms returned by the
   *                    {@link #getSASLMechanismNames} method.
   *
   * @return  {@code true} if the specified SASL mechanism should be considered
   *          secure, or {@code false} if not.
   */
  public abstract boolean isSecure(final String mechanism);



  /**
   * Indicates whether the SASL authentication process using the specified
   * mechanism involves the use of a password stored locally in the server
   * (optionally in combination with other forms of credentials).
   *
   * @param  mechanism  The name of the mechanism for which to make the
   *                    determination.  This will only be invoked with names of
   *                    mechanisms returned by the
   *                    {@link #getSASLMechanismNames} method.
   *
   * @return  {@code true} if the specified SASL mechanism makes use of a local
   *          password, or {@code false} if not.
   */
  public abstract boolean isPasswordBased(final String mechanism);



  /**
   * Performs the appropriate processing for the provided SASL bind request.
   *
   * @param  operationContext  The context for the bind operation.
   * @param  bindRequest       The SASL bind request to be processed.
   * @param  resultFactory     A factory object that will be used to construct
   *                           the result to return.
   *
   * @return  An object with information about the result of the SASL bind
   *          processing.
   */
  public abstract SASLBindResult processSASLBind(
                       final OperationContext operationContext,
                       final SASLBindRequest bindRequest,
                       final SASLBindResultFactory resultFactory);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }



  /**
   * Indicates whether the specified SASL mechanism, which is listed as
   * supported by this handler, is available for use by the given user.
   *
   * @param  mechanism          The name of the SASL mechanism for which to make
   *                            the determination.  It will be one of the
   *                            mechanisms for which this handler is registered.
   * @param  passwordAttribute  The name of the attribute used to hold the
   *                            password for the user.
   * @param  userEntry          The entry for the user for whom to make the
   *                            determination.
   *
   * @return  {@code true} if the SASL mechanism is supported for the user,
   *          {@code false} if not, or {@code null} if it could not be
   *          determined.
   */
  public Boolean isMechanismAvailableForUser(final String mechanism,
                                             final String passwordAttribute,
                                             final Entry userEntry)
  {
    // This should be overridden by subclasses that wish to implement this
    // functionality.
    return null;
  }



  /**
   * Indicates whether the specified SASL mechanism may require multiple stages
   * to process.
   *
   * @param  mechanism  The mechanism for which to make the determination.
   *
   * @return  {@code true} if the specified SASL mechanism may require multiple
   *          stages to process, {@code false} if not, or {@code null} if the
   *          answer is not known for the specified mechanism.
   */
  public Boolean isMultiStageMechanism(final String mechanism)
  {
    // This should be overridden by subclasses that wish to impelement this
    // functionality.
    return null;
  }
}
