/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2013-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.ds.config.SearchEntryCriteriaConfig;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.Control;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which can
 * be used to classify client search result entries.
 * <BR>
 * <H2>Configuring Search Entry Criteria</H2>
 * In order to configure a search entry criteria object created using this API,
 * use a command like:
 * <PRE>
 *      dsconfig create-search-entry-criteria \
 *           --criteria-name "<I>{criteria-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{criteria-name}</I>" is the name to use for the search entry
 * criteria instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.ds.api.SearchEntryCriteria}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the search entry criteria.  If multiple arguments should be
 * provided to the search entry criteria instance, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class SearchEntryCriteria
       implements UnboundIDExtension,
                  Reconfigurable<SearchEntryCriteriaConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this search entry criteria.  All search entry
   * criteria implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeSearchEntryCriteria} method.
   */
  public SearchEntryCriteria()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this search entry criteria.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this search entry
   *                        criteria.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this search entry criteria.
   *
   * @throws  LDAPException  If a problem occurs while initializing this search
   *                         entry criteria.
   */
  public void initializeSearchEntryCriteria(
                   final DirectoryServerContext serverContext,
                   final SearchEntryCriteriaConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final SearchEntryCriteriaConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final SearchEntryCriteriaConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this search entry " +
                "criteria is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this search entry criteria
   * is to be taken out of service.
   */
  public void finalizeSearchEntryCriteria()
  {
    // No implementation is required.
  }



  /**
   * Indicates whether the provided operation matches the criteria for this
   * search entry criteria object.
   *
   * @param  o         The context for the associated search operation.
   * @param  entry     The entry for which to make the determination.
   * @param  controls  The controls included with the search result entry.
   *
   * @return  {@code true} if the provided operation matches the criteria for
   *          this search entry criteria object, or {@code false} if not.
   */
  public abstract boolean matches(final OperationContext o, final Entry entry,
                                  final List<Control> controls);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
