/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.ds.types.TaskContext;
import com.unboundid.directory.sdk.ds.types.TaskReturnState;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * may be used as administrative tasks.  Administrative tasks have the ability
 * to perform arbitrary processing within the server whenever a particular kind
 * of entry is added, and that processing can be performed immediately or at
 * some specified time in the future.  The server must be configured as follows
 * to allow creation of third-party tasks:
 * <PRE>
 *   dsconfig set-global-configuration-prop
 *     --add allowed-task:com.unboundid.directory.sdk.extensions.ThirdPartyTask
 * </PRE>
 * Tasks may then be scheduled by adding an entry
 * below "cn=Scheduled Tasks,cn=tasks" with a format like the following:
 * <PRE>
 *   dn:  ds-task-id=TASKID,cn=Scheduled Tasks,cn=tasks
 *   objectClass: top
 *   objectClass: ds-task
 *   objectClass: ds-third-party-task
 *   ds-task-id: TASKID
 *   ds-task-class-name: com.unboundid.directory.sdk.extensions.ThirdPartyTask
 *   ds-third-party-task-java-class: com.example.ExampleTask
 *   ds-third-party-task-argument: name=value
 * </PRE>
 * In this example, TASKID should be replaced with a string that uniquely
 * identifies the task.  The value of the ds-third-party-task-java-class
 * attribute should contain the fully-qualified name of the non-abstract Java
 * class that extends this com.unboundid.directory.sdk.api.Task class, and the
 * ds-third-party-task-argument values (if any) should reflect the set of
 * arguments allowed for the task as per the {@link #defineConfigArguments}
 * method.
 * <BR><BR>
 * Alternately, the com.unboundid.ldap.sdk.unboundidds.tasks.ThirdPartyTask
 * class included in the Commercial Edition of the UnboundID LDAP SDK for Java
 * may be used to schedule and interact with these kinds of tasks.  See the
 * documentation for the Commercial Edition of the LDAP SDK for more
 * information on using the UnboundID LDAP SDK for Java to schedule and interact
 * with administrative tasks.
 *
 * @see  com.unboundid.directory.sdk.ds.scripting.ScriptedTask
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class Task
       implements UnboundIDExtension, ExampleUsageProvider
{
  /**
   * Creates a new instance of this task.  All task implementations must include
   * a default constructor, but any initialization should generally be done in
   * the {@code initializeTask} method.
   */
  public Task()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this task.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this task is running.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this task.
   *
   * @throws  LDAPException  If a problem occurs while initializing this task.
   */
  public void initializeTask(final DirectoryServerContext serverContext,
                             final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * Retrieves a human-readable name that may be used for this task.
   *
   * @return  A human-readable name that may be used for this task.
   */
  public abstract String getTaskName();



  /**
   * Performs the appropriate processing for this task.
   *
   * @param  taskContext  Information about the task to be run.
   *
   * @return  Information about the state of the task after processing has
   *          completed.
   */
  public abstract TaskReturnState runTask(final TaskContext taskContext);



  /**
   * Indicates whether this task may be interrupted before it has completed
   * (e.g., canceled by an administrator or aborted at server shutdown).  It is
   * particularly important that potentially long-running tasks be interruptible
   * so that they do not impede server shutdown or consume excessive resources.
   *
   * @return  {@code true} if this task may be interrupted before it has
   *          completed, or {@code false} if it cannot be interrupted.
   */
  public boolean isInterruptible()
  {
    return false;
  }



  /**
   * Attempts to interrupt the execution of this task.  This should only be
   * called if the {@link #isInterruptible} method returns {@code true}.
   *
   * @param  interruptState   The return state that should be used for the task
   *                          if it is successfully interrupted.
   * @param  interruptReason  A message that provides a reason that the task has
   *                          been interrupted.
   */
  public void interruptTask(final TaskReturnState interruptState,
                            final String interruptReason)
  {
    // No action is performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
