/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.ds.config.VirtualAttributeProviderConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.scripting.
            ScriptedVirtualAttributeProvider;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * construct attribute values which may be included in entries instead of or
 * in addition to real values which are actually stored in the backend.  The
 * other attributes in the entry will be available for use in the process of
 * generating the entry, and internal or external operations may also be
 * performed if the generated values should incorporate data from other
 * locations.
 * <BR><BR>
 * Each virtual attribute provider may be configured to indicate whether the
 * associated virtual attribute should be included in a given entry.  This
 * criteria may include the entry's location in the DIT, whether it matches a
 * given filter, whether it is a member of a specified group, and whether the
 * requesting client has been assigned a given client connection policy.  This
 * is handled automatically by the server, so individual virtual attribute
 * provider implementations do not need to attempt to perform that filtering on
 * their own.  However, they may perform additional processing if desired to
 * further narrow the set of entries for which the virtual attribute should be
 * generated.
 * <BR><BR>
 * In addition, virtual attribute providers may be configured to indicate the
 * behavior that should be exhibited in the event that the target attribute
 * already exists in the entry with one or more real values.  In this case, the
 * real values may be used instead of generating virtual values, the virtual
 * values may be used in place of the real values, or both the real and virtual
 * values may be merged and presented together.  This work is also automatically
 * performed by the server, so virtual attribute providers do not need to do any
 * processing to determine whether to generate a value based on whether the
 * target attribute already exists in the entry.
 * <BR><BR>
 * The server supports multiple virtual attribute providers targeting the same
 * attribute applying to the same entry. Evaluation order and value selection is
 * determined by the server based on configuration of the virtual attribute
 * providers.
 * <BR>
 * <H2>Configuring Virtual Attribute Providers</H2>
 * In order to configure a virtual attribute provider created using this API,
 * use a command like:
 * <PRE>
 *      dsconfig create-virtual-attribute \
 *           --name "<I>{name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set attribute-type:{attribute} \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{name}</I>" is the name to use for the virtual attribute provider
 * instance, "<I>{attribute}</I>" is the name of the attribute for which the
 * virtual attribute provider should be used to generate values,
 * "<I>{class-name}</I>" is the fully-qualified name of the Java class that
 * extends {@code com.unboundid.directory.sdk.ds.api.VirtualAttributeProvider},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the virtual attribute provider.  If multiple arguments should be
 * provided to the virtual attribute provider, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  ScriptedVirtualAttributeProvider
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class VirtualAttributeProvider
       implements UnboundIDExtension,
                  Reconfigurable<VirtualAttributeProviderConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this virtual attribute provider.  All virtual
   * attribute provider implementations must include a default constructor, but
   * any initialization should generally be done in the
   * {@code initializeVirtualAttributeProvider} method.
   */
  public VirtualAttributeProvider()
  {
    // No implementation is required.  However, we need to reference the
    // scripted virtual attribute provider so that checkstyle is satisfied with
    // the import.
    final ScriptedVirtualAttributeProvider scriptedProvider = null;
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this virtual attribute provider.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this virtual attribute
   *                        provider.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this virtual attribute
   *                        provider.
   *
   * @throws  LDAPException  If a problem occurs while initializing this virtual
   *                         attribute provider.
   */
  public void initializeVirtualAttributeProvider(
                   final DirectoryServerContext serverContext,
                   final VirtualAttributeProviderConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final VirtualAttributeProviderConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(
                         final VirtualAttributeProviderConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this virtual " +
                "attribute provider is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this virtual attribute
   * provider is to be taken out of service.
   */
  public void finalizeVirtualAttributeProvider()
  {
    // No implementation is required.
  }



  /**
   * Indicates whether the server may cache values generated by this virtual
   * attribute provider for reuse against the same entry in the course of
   * processing the same operation.
   *
   * @return  {@code true} if the server may cache the value generated by this
   *          virtual attribute provider for reuse with the same entry in the
   *          same operation, or {@code false} if not.
   */
  public boolean mayCacheInOperation()
  {
    return false;
  }



  /**
   * Indicates whether this virtual attribute provider may generate attributes
   * with multiple values.
   *
   * @return  {@code true} if this virtual attribute provider may generate
   *          attributes with multiple values, or {@code false} if it will only
   *          generate single-valued attributes.
   */
  public abstract boolean isMultiValued();



  /**
   * Generates an attribute for inclusion in the provided entry.
   *
   * @param  operationContext  The operation context for the operation in
   *                           progress, if any.  It may be {@code null} if no
   *                           operation is available.
   * @param  entry             The entry for which the attribute is to be
   *                           generated.
   * @param  attributeName     The name of the attribute to be generated.
   *
   * @return  The generated attribute, or {@code null} if no attribute should be
   *          generated.
   */
  public abstract Attribute generateAttribute(
                                 final OperationContext operationContext,
                                 final Entry entry, final String attributeName);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
