/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.scripting;



import java.util.List;

import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.ds.config.
            AccountStatusNotificationHandlerConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.AccountStatusNotification;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by scripted extensions
 * which have the ability to process notifications about events that occur that
 * could change the status of a user account.  For example, account status
 * notification handlers may be used to notify the end user and/or
 * administrators of a change to that user's account.  The kinds of events that
 * may trigger account status notifications are defined in the
 * {@link com.unboundid.directory.sdk.ds.types.AccountStatusNotificationType}
 * enum, and include things like user accounts locked due to authentication
 * failures, password expiration errors and warnings, and administrative
 * password resets.
 * <BR>
 * <H2>Configuring Groovy-Scripted Account Status Notification Handlers</H2>
 * In order to configure a scripted account status notification handler based on
 * this API and written in the Groovy scripting language, use a command like:
 * <PRE>
 *      dsconfig create-account-status-notification-handler \
 *           --handler-name "<I>{handler-name}</I>" \
 *           --type groovy-scripted \
 *           --set enabled:true \
 *           --set "script-class:<I>{class-name}</I>" \
 *           --set "script-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{handler-name}</I>" is the name to use for the account status
 * notification handler instance, "<I>{class-name}</I>" is the fully-qualified
 * name of the Groovy class written using this API, and "<I>{name=value}</I>"
 * represents name-value pairs for any arguments to provide to the account
 * status notification handler.  If multiple arguments should be provided to the
 * account status notification handler, then the
 * "<CODE>--set script-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.ds.api.AccountStatusNotificationHandler
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ScriptedAccountStatusNotificationHandler
       implements Reconfigurable<AccountStatusNotificationHandlerConfig>
{
  /**
   * Creates a new instance of this account status notification handler.  All
   * account status notification handler implementations must include a default
   * constructor, but any initialization should generally be done in the
   * {@code initializeAccountStatusNotificationHandler} method.
   */
  public ScriptedAccountStatusNotificationHandler()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this account status notification handler.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this account status
   *                        notification handler.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this account status
   *                        notification handler.
   *
   * @throws  LDAPException  If a problem occurs while initializing this account
   *                         status notification handler.
   */
  public void initializeAccountStatusNotificationHandler(
                   final DirectoryServerContext serverContext,
                   final AccountStatusNotificationHandlerConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final AccountStatusNotificationHandlerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(
                         final AccountStatusNotificationHandlerConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this account " +
                "status notification handler is disabled and re-enabled or " +
                "until the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this account status
   * notification handler is to be taken out of service.
   */
  public void finalizeAccountStatusNotificationHandler()
  {
    // No implementation is required.
  }



  /**
   * Performs any processing that may be necessary in conjunction with the
   * provided account status notification.
   *
   * @param  notification  The account status notification to be processed.
   */
  public abstract void handleStatusNotification(
                            final AccountStatusNotification notification);
}
