/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.scripting;



import java.security.cert.Certificate;
import java.util.List;

import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.ds.config.CertificateMapperConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by scripted extensions
 * which attempt to map a certificate presented by a client (e.g., via SSL or
 * StartTLS) to a user defined in the server.  This is primarily used during
 * the course of SASL EXTERNAL bind processing when a client uses a certificate
 * to authenticate to the server.  Any information contained in the provided
 * certificate chain (including the subject, fingerprint, validity, extensions,
 * etc. of the client certificate, as well as any information about any issuer
 * certificates) may be used to map information in the provided certificate
 * chain to exactly one user in the server.  If the certificate cannot be mapped
 * to any user, or if it is mapped to multiple users, then the authentication
 * attempt must fail.
 * <BR>
 * <H2>Configuring Groovy-Scripted Certificate Mappers</H2>
 * In order to configure a scripted certificate mapper based on this API and
 * written in the Groovy scripting language, use a command like:
 * <PRE>
 *      dsconfig create-certificate-mapper \
 *           --mapper-name "<I>{mapper-name}</I>" \
 *           --type groovy-scripted \
 *           --set enabled:true \
 *           --set "script-class:<I>{class-name}</I>" \
 *           --set "script-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{mapper-name}</I>" is the name to use for the certificate mapper
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Groovy
 * class written using this API, and "<I>{name=value}</I>" represents name-value
 * pairs for any arguments to provide to the certificate mapper.  If multiple
 * arguments should be provided to the certificate mapper, then the
 * "<CODE>--set script-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.ds.api.CertificateMapper
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=false)
@SynchronizationServerExtension(appliesToLocalContent=true,
     appliesToSynchronizedContent=false)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ScriptedCertificateMapper
       implements Reconfigurable<CertificateMapperConfig>
{
  /**
   * Creates a new instance of this certificate mapper.  All certificate mapper
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@code initializeCertificateMapper} method.
   */
  public ScriptedCertificateMapper()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this certificate mapper.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this certificate
   *                        mapper.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this certificate mapper.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         certificate mapper.
   */
  public void initializeCertificateMapper(
                   final DirectoryServerContext serverContext,
                   final CertificateMapperConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final CertificateMapperConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final CertificateMapperConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this certificate " +
                "mapper is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this certificate mapper is
   * to be taken out of service.
   */
  public void finalizeCertificateMapper()
  {
    // No implementation is required.
  }



  /**
   * Performs any processing which may be necessary to map the provided
   * certificate chain to a user within the server.
   *
   * @param  certChain  The certificate chain presented by the client.
   *
   * @return  The DN of the user within the server to which the provided
   *          certificate corresponds.
   *
   * @throws  LDAPException  If the presented certificate cannot be mapped to
   *                         exactly one user in the server.
   */
  public abstract String mapCertificate(final Certificate[] certChain)
         throws LDAPException;
}
