/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2012-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.scripting;



import java.util.List;

import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.directory.sdk.ds.config.UncachedAttributeCriteriaConfig;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.ds.types.DirectoryServerContext;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by scripted extensions
 * which have the ability to determine which attributes should be stored in the
 * uncached-id2entry database of a local DB backend, while the remainder of the
 * entry is stored in the id2entry database.  In environments with data sets too
 * large to fit in available memory, and in which entries have a significant
 * amount of space consumed by attributes which are not frequently accessed,
 * this can help the server better use the memory it does have for attributes
 * that are more likely to be accessed.
 * <BR>
 * <H2>Configuring Groovy-Scripted Uncached Attribute Criteria</H2>
 * In order to configure a scripted uncached attribute criteria object based on
 * this API and written in the Groovy scripting language, use a command like:
 * <PRE>
 *      dsconfig create-uncached-attribute-criteria \
 *           --criteria-name "<I>{criteria-name}</I>" \
 *           --type groovy-scripted \
 *           --set enabled:true \
 *           --set "script-class:<I>{class-name}</I>" \
 *           --set "script-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{handler-name}</I>" is the name to use for the uncached attribute
 * criteria instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Groovy class written using this API, and "<I>{name=value}</I>" represents
 * name-value pairs for any arguments to provide to the uncached attribute
 * criteria object.  If multiple arguments should be provided to the uncached
 * attribute criteria, then the
 * "<CODE>--set script-argument:<I>{name=value}</I></CODE>"
 * option should be provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.ds.api.UncachedAttributeCriteria
 */
@Extensible()
@DirectoryServerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ScriptedUncachedAttributeCriteria
       implements Reconfigurable<UncachedAttributeCriteriaConfig>
{
  /**
   * Creates a new instance of this uncached attribute criteria.  All uncached
   * attribute criteria implementations must include a default constructor, but
   * any initialization should generally be done in the
   * {@code initializeUncachedAttributeCriteria} method.
   */
  public ScriptedUncachedAttributeCriteria()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this uncached attribute criteria.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this uncached
   *                        attribute criteria.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this uncached attribute
   *                        criteria.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         uncached attribute criteria.
   */
  public void initializeUncachedAttributeCriteria(
                   final DirectoryServerContext serverContext,
                   final UncachedAttributeCriteriaConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final UncachedAttributeCriteriaConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(
                         final UncachedAttributeCriteriaConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this uncached " +
                "attribute criteria is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this uncached attribute
   * criteria instance is to be taken out of service.
   */
  public void finalizeUncachedAttributeCriteria()
  {
    // No implementation is required.
  }



  /**
   * Indicates whether the provided attribute should be written into the
   * uncached-id2entry database rather than into id2entry.
   *
   * @param  attribute  A read-only representation of the attribute for which to
   *                    make the determination.
   * @param  entry      A read-only representation of the full entry to be
   *                    encoded.
   *
   * @return  {@code true} if the attribute should be written into the
   *          uncached-id2entry database, or {@code false} if it should be
   *          written into the id2entry database.
   */
  public abstract boolean shouldBeUncached(final Attribute attribute,
                                           final Entry entry);
}
