/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;

import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

/**
 * This interface defines a set of methods that may be used to interact with a
 * backend that has been defined in the server.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface BackendContext
{

  /**
   * Retrieves the backend ID for this backend.
   *
   * @return  The backend ID for this backend.
   */
  String getBackendID();



  /**
   * Retrieves the set of base DNs handled by the backend.
   *
   * @return  The set of base DNs handled by the backend.
   */
  String[] getBaseDNs();



  /**
   * Retrieves the total number of entries contained in this backend,
   * if that information is available.
   *
   * @return  The total number of entries contained in this backend,
   *          or -1 if that information is not available.
   */
  long getEntryCount();



  /**
   * Indicates whether the backend is considered local (i.e., holds data in a
   * repository that is local to and entirely managed by the server).
   *
   * @return  {@code true} if the backend should be considered local, or
   *          {@code false} if the data may be held in some external repository.
   */
  boolean isLocal();



  /**
   * Indicates whether the backend is considered a private backend (with data
   * generated and/or maintained by the server) rather than a public backend
   * (with user data provided and managed by clients).
   *
   * @return  {@code true} if the backend is considered a private backend within
   *          the server, or {@code false} if it is a public backend.
   */
  boolean isPrivateBackend();



  /**
   * Retrieves the {@link WritabilityMode} for the backend.
   *
   * @return  The {@link WritabilityMode} for the backend.
   */
  WritabilityMode getWritabilityMode();



  /**
   * Indicates whether the backend considers filter components involving the
   * specified attribute type and index type indexed. This may include
   * attribute types for which an index has been explicitly configured in the
   * backend, as well as attributes which are automatically considered indexed
   * (e.g., because the backend holds only a small amount of data and considers
   * all attributes indexed).
   *
   * @param  attributeType  The name or OID of the attribute type for which to
   *                        make the determination. It must not be
   *                        {@code null}.
   * @param  indexType      The {@link IndexType} for which to make the
   *                        determination. It must not be {@code null}.
   *
   * @return  {@code true} if the specified attribute type is indexed for the
   *          given type of matching, or {@code false} if not.
   */
  boolean isIndexed(final String attributeType, final IndexType indexType);



  /**
   * Indicates whether the backend has been explicitly configured with an
   * index for the specified attribute type and index type.
   *
   * @param  attributeType  The name or OID of the attribute type for which to
   *                        make the determination. It must not be
   *                        {@code null}.
   * @param  indexType      The {@link IndexType} for which to make the
   *                        determination. It must not be {@code null}.
   *
   * @return  {@code true} if the specified attribute type is indexed for the
   *          given type of matching, or {@code false} if not.
   */
  boolean isExplicitlyIndexed(final String attributeType,
                              final IndexType indexType);



  /**
   * Indicates whether the backend may consider a search with the provided
   * filter as indexed.
   *
   * @param  filter  The {@link Filter} for which to make the determination.
   *                 It must not be {@code null}.
   *
   * @return  {@code true} if a search with the provided filter may be
   *          considered indexed in the backend, or {@code false} if not.
   */
  boolean isIndexed(final Filter filter);



  /**
   * Indicates whether the specified entry exists in the backend.
   *
   * @param  dn  The DN for which to make the determination. It must not be
   *             {@code null}.
   *
   * @return  {@code true} if the specified entry exists in the backend, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If the provided string cannot be parsed as a valid
   *                         DN or if a problem is encountered while making the
   *                         determination.
   */
  boolean entryExists(final String dn) throws LDAPException;



  /**
   * Retrieves the specified entry from the backend.
   *
   * @param  dn  The DN for which to make the determination. It must not be
   *             {@code null}.
   *
   * @return  The entry with the specified DN, or {@code null} if the entry does
   *          not exist in the backend.
   *
   * @throws  LDAPException  If the provided string cannot be parsed as a valid
   *                         DN or if a problem is encountered while attempting
   *                         to retrieve it.
   */
  Entry getEntry(final String dn) throws LDAPException;



  /**
   * Indicates whether this backend should contain the entry with the specified
   * DN.
   *
   * @param dn  The DN of the entry for which to make the determination.
   *
   * @return  {@code true} if this backend handles operations for the
   *          provided entry, or {@code false} if it does not.
   *
   * @throws  LDAPException  If the provided string cannot be parsed as a valid
   *                         DN or if a problem is encountered while making the
   *                         determination.
   */
   boolean handlesEntry(String dn) throws LDAPException;



  /**
   * Retrieves information about the parent for the associated backend, if any.
   *
   * @return  Information about the parent for the associated backend, or
   *          {@code null} if the backend does not have a parent.
   */
  BackendContext getParentBackend();



  /**
   * Retrieves information about any backends which are immediately subordinate
   * to the associated backend.
   *
   * @return  Information about any backends which are immediately subordinate
   *          to the associated backend, or an empty array if there are no
   *          subordinate backends.
   */
  BackendContext[] getSubordinateBackends();
}
