/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import com.unboundid.directory.sdk.common.types.ServerContext;
import com.unboundid.directory.sdk.ds.api.BackendInitializationListener;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface may be used to obtain information about the Directory Server
 * (or related product) in which an extension is running.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public interface DirectoryServerContext extends ServerContext
{
  /**
   * Returns an array of all the backends defined in the Directory Server.
   *
   * @return an array of {@link BackendContext} objects. This will never
   *         be null.
   */
  BackendContext[] getAllBackends();



  /**
   * Returns the {@link BackendContext} for the backend which should handle the
   * specified entry, or null if there is no backend that handles this entry.
   * Note that the entry does not need to exist. If a BackendContext is
   * returned, it will return true for
   * {@link BackendContext#handlesEntry(String)} when given the specified DN.
   *
   * @param dn The DN for which to make the determination. It must not be
   *             {@code null}.
   *
   * @return The {@link BackendContext} for the given entry, or null if no such
   *         backend exists.
   *
   * @throws  LDAPException  If the provided string cannot be parsed as a valid
   *                         DN or if a problem is encountered while attempting
   *                         to retrieve the BackendContext.
   */
  BackendContext getBackendForEntry(String dn) throws LDAPException;



  /**
   * Registers the provided backend initialization listener with the server so
   * that it may be notified of any backends coming online or going offline.
   *
   * @param  listener     The backend initialization listener to be registered
   *                      with the server. It must not be {@code null}.
   *
   * @return  An object representing the backend initialization listener that
   *          has been registered with the server.
   */
  RegisteredBackendInitializationListener registerBackendInitializationListener(
                      final BackendInitializationListener listener);



  /**
   * Deregisters the provided backend initialization listener with the server.
   * This will have no effect if the provided listener is not registered.
   *
   * @param  listener  The listener to be deregistered.  It must not be
   *                   {@code null}.
   */
  void deregisterBackendInitializationListener(
          final RegisteredBackendInitializationListener listener);
}
