/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import java.io.Serializable;

import com.unboundid.util.NotMutable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This class defines a structure which may be used to provide information about
 * the result of the processing performed by an intermediate response plugin.
 */
@NotMutable()
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class IntermediateResponsePluginResult
       implements Serializable
{
  /**
   * A predefined result instance that indicates all processing completed
   * successfully.
   */
  public static final IntermediateResponsePluginResult SUCCESS =
       new IntermediateResponsePluginResult(false, true, true, true);



  /**
   * The serial version UID for this serializable class.
   */
  private static final long serialVersionUID = -4104659694213854665L;



  // Indicates whether the client connection was terminated by the plugin.
  private final boolean connectionTerminated;

  // Indicates whether the server should continue processing other
  // intermediate response plugins for the associated response.
  private final boolean continuePluginProcessing;

  // Indicates whether the server should continue processing for the associated
  // operation.
  private final boolean continueOperationProcessing;

  // Indicates whether the associated intermediate response should be returned
  // to the client.
  private final boolean sendIntermediateResponse;



  /**
   * Creates a new intermediate response plugin result with the provided
   * information.
   *
   * @param  connectionTerminated         Indicates whether the client
   *                                      connection was terminated by the
   *                                      plugin.
   * @param  continuePluginProcessing     Indicates whether to continue
   *                                      processing other intermediate response
   *                                      plugins for the associated response.
   * @param  sendIntermediateResponse     Indicates whether to send the
   *                                      intermediate response to the client.
   * @param  continueOperationProcessing  Indicates whether to continue
   *                                      processing the associated operation.
   */
  public IntermediateResponsePluginResult(final boolean connectionTerminated,
              final boolean continuePluginProcessing,
              final boolean sendIntermediateResponse,
              final boolean continueOperationProcessing)
  {
    this.connectionTerminated        = connectionTerminated;
    this.continuePluginProcessing    = continuePluginProcessing;
    this.continueOperationProcessing = continueOperationProcessing;
    this.sendIntermediateResponse    = sendIntermediateResponse;
  }



  /**
   * Indicates whether the client connection was terminated by the plugin.
   *
   * @return  {@code true} if the client connection was terminated by the
   *          plugin, or {@code false} if not.
   */
  public boolean connectionTerminated()
  {
    return connectionTerminated;
  }



  /**
   * Indicates whether to continue processing other intermediate response
   * plugins for the associated response.
   *
   * @return  {@code true} if the server should continue processing other
   *          intermediate response plugins for the associated response, or
   *          {@code false} if not.
   */
  public boolean continuePluginProcessing()
  {
    return continuePluginProcessing;
  }



  /**
   * Indicates whether the associated intermediate response should be returned
   * to the client.
   *
   * @return  {@code true} if the intermediate response should be returned to
   *          the client, or {@code false} if the server should not return the
   *          intermediate response to the client.
   */
  public boolean sendIntermediateResponse()
  {
    return sendIntermediateResponse;
  }



  /**
   * Indicates whether to continue processing for the associated operation.
   *
   * @return  {@code true} if the server should continue processing for the
   *          associated operation, or {@code false} if not and the operation
   *          response should be sent to the client.
   */
  public boolean continueOperationProcessing()
  {
    return continueOperationProcessing;
  }



  /**
   * Retrieves a string representation of this intermediate response plugin
   * result.
   *
   * @return  A string representation of this intermediate response plugin
   *          result.
   */
  @Override()
  public String toString()
  {
    final StringBuilder buffer = new StringBuilder();

    buffer.append("IntermediateResponsePluginResult(connectionTerminated=");
    buffer.append(connectionTerminated);
    buffer.append(", continuePluginProcessing=");
    buffer.append(continuePluginProcessing);
    buffer.append(", sendIntermediateResponse=");
    buffer.append(sendIntermediateResponse);
    buffer.append(", continueOperationProcessing=");
    buffer.append(continueOperationProcessing);
    buffer.append(')');

    return buffer.toString();
  }
}
