/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import java.io.Serializable;

import com.unboundid.util.NotMutable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This class defines a structure which may be used to provide information about
 * the result of the processing performed by an LDIF import or export plugin.
 */
@NotMutable()
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class LDIFPluginResult
       implements Serializable
{
  /**
   * A predefined result instance that indicates all processing completed
   * successfully.
   */
  public static final LDIFPluginResult SUCCESS =
       new LDIFPluginResult(true, true, null);



  /**
   * The serial version UID for this serializable class.
   */
  private static final long serialVersionUID = -5261599290257098170L;



  // Indicates whether the server should continue processing other LDIF import
  // or export plugins.
  private final boolean continuePluginProcessing;

  // Indicates whether the associated entry should continue to be included in
  // the LDIF import or export.
  private final boolean includeEntry;

  // A message providing additional information about the reason the entry
  // should be excluded.
  private final String excludeReason;



  /**
   * Creates a new LDIF plugin result with the provided information.
   *
   * @param  continuePluginProcessing  Indicates whether to continue processing
   *                                   other LDIF import or export plugins.
   * @param  includeEntry              Indicates whether the associated entry
   *                                   should continue to be included in the
   *                                   LDIF import or export.
   * @param  excludeReason             An optional message providing additional
   *                                   information about the reason the entry
   *                                   should be excluded from the import or
   *                                   export.  It may be {@code null} if the
   *                                   entry should still be included.
   */
  public LDIFPluginResult(final boolean continuePluginProcessing,
                          final boolean includeEntry,
                          final CharSequence excludeReason)
  {
    this.continuePluginProcessing = continuePluginProcessing;
    this.includeEntry             = includeEntry;

    if (excludeReason == null)
    {
      this.excludeReason = null;
    }
    else
    {
      this.excludeReason = excludeReason.toString();
    }
  }



  /**
   * Indicates whether to continue processing other LDIF import or export
   * plugins for the entry.
   *
   * @return  {@code true} if the server should continue processing other LDIF
   *          import or export plugins for the entry, or {@code false} if not.
   */
  public boolean continuePluginProcessing()
  {
    return continuePluginProcessing;
  }



  /**
   * Indicates whether the associated entry should be still be included in the
   * LDIF import or export.
   *
   * @return  {@code true} if the associated entry should be still be included
   *          in the LDIF import or export, or {@code false} if it should be
   *          excluded.
   */
  public boolean includeEntry()
  {
    return includeEntry;
  }



  /**
   * Retrieves a message with additional information about the reason the entry
   * should be excluded, if applicable.
   *
   * @return  A message with additional information about the reason the entry
   *          should be excluded, or {@code null} if none is available or the
   *          entry should be included in the import or export.
   */
  public String getExcludeReason()
  {
    return excludeReason;
  }



  /**
   * Retrieves a string representation of this LDIF plugin result.
   *
   * @return  A string representation of this LDIF plugin result.
   */
  @Override()
  public String toString()
  {
    final StringBuilder buffer = new StringBuilder();

    buffer.append("LDIFPluginResult(continuePluginProcessing=");
    buffer.append(continuePluginProcessing);
    buffer.append(", includeEntry=");
    buffer.append(includeEntry);

    if (excludeReason != null)
    {
      buffer.append(", excludeReason='");
      buffer.append(excludeReason);
      buffer.append('\'');
    }

    buffer.append(')');

    return buffer.toString();
  }
}
