/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2014-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import com.unboundid.directory.sdk.common.schema.AttributeType;
import com.unboundid.util.NotMutable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;



/**
 * This class defines a structure used by a notification manager to specify
 * information that it wishes to be included in a change notification.
 */
@NotMutable()
@ThreadSafety(level= ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class NotificationProperties
{
  private final String              destinationID;
  private final Map<String,String>  propertyMap;
  private final Set<AttributeType>  keyAttributes;



  /**
   * Create a new instance of this class from the provided information.
   *
   * @param destinationID  The notification destination ID.
   * @param propertyMap    A set of property names and values to be included
   *                       in the notification. May be {@code null}.
   * @param keyAttributes  Any additional key attributes to be recorded in the
   *                       LDAP changelog entry and included in the
   *                       notification. May be {@code null}.
   */
  public NotificationProperties(
      final String destinationID,
      final Map<String,String> propertyMap,
      final Set<AttributeType> keyAttributes
  )
  {
    this.destinationID = destinationID;

    if (propertyMap == null)
    {
      this.propertyMap = Collections.emptyMap();
    }
    else
    {
      this.propertyMap = Collections.unmodifiableMap(
          new HashMap<String, String>(propertyMap));
    }

    if (keyAttributes == null)
    {
      this.keyAttributes = Collections.emptySet();
    }
    else
    {
      this.keyAttributes = Collections.unmodifiableSet(
          new HashSet<AttributeType>(keyAttributes));
    }
  }



  /**
   * Retrieve the notification destination ID.
   *
   * @return  The notification destination ID.
   */
  public String getDestinationID()
  {
    return destinationID;
  }



  /**
   * Retrieve the set of property names and values to be included in the
   * notification.
   *
   * @return  The set of property names and values to be included in the
   *          notification. Guaranteed to be non-null.
   */
  public Map<String, String> getPropertyMap()
  {
    return propertyMap;
  }



  /**
   * Retrieve the additional key attributes to be recorded in the LDAP
   * changelog entry and included in the notification.
   *
   * @return  The additional key attributes to be recorded in the LDAP
   *          changelog entry and included in the notification.
   */
  public Set<AttributeType> getKeyAttributes()
  {
    return keyAttributes;
  }
}
