/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import java.io.Serializable;

import com.unboundid.util.NotMutable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This class defines a structure which may be used to provide information about
 * the result of the processing performed by a post-connect plugin.
 */
@NotMutable()
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class PostConnectPluginResult
       implements Serializable
{
  /**
   * A predefined result instance that indicates all processing completed
   * successfully.
   */
  public static final PostConnectPluginResult SUCCESS =
       new PostConnectPluginResult(false, true);



  /**
   * The serial version UID for this serializable class.
   */
  private static final long serialVersionUID = 3975156371629164031L;



  // Indicates whether the connection has been terminated.
  private final boolean connectionTerminated;

  // Indicates whether the server should continue processing other post-connect
  // plugins.
  private final boolean continuePluginProcessing;



  /**
   * Creates a new post-connect plugin result with the provided information.
   *
   * @param  connectionTerminated      Indicates whether the connection has been
   *                                   terminated.
   * @param  continuePluginProcessing  Indicates whether to continue processing
   *                                   other post-connect plugins for the
   *                                   connection.
   */
  public PostConnectPluginResult(final boolean connectionTerminated,
                                 final boolean continuePluginProcessing)
  {
    this.connectionTerminated     = connectionTerminated;
    this.continuePluginProcessing = continuePluginProcessing;
  }



  /**
   * Indicates whether the connection has been terminated.
   *
   * @return  {@code true} if the connection has been terminated, or
   *          {@code false} if not.
   */
  public boolean connectionTerminated()
  {
    return connectionTerminated;
  }



  /**
   * Indicates whether to continue processing other post-connect plugins for the
   * connection.
   *
   * @return  {@code true} if the server should continue processing other
   *          post-connect plugins for the connection, or {@code false} if not.
   */
  public boolean continuePluginProcessing()
  {
    return continuePluginProcessing;
  }



  /**
   * Retrieves a string representation of this post-connect plugin result.
   *
   * @return  A string representation of this post-connect plugin result.
   */
  @Override()
  public String toString()
  {
    final StringBuilder buffer = new StringBuilder();

    buffer.append("PostConnectPluginResult(connectionTerminated=");
    buffer.append(connectionTerminated);
    buffer.append(", continuePluginProcessing=");
    buffer.append(continuePluginProcessing);
    buffer.append(')');

    return buffer.toString();
  }
}
