/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import java.io.Serializable;

import com.unboundid.util.NotMutable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This class defines a structure which may be used to provide information about
 * the result of the processing performed by a pre-operation plugin.
 */
@NotMutable()
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class PreOperationPluginResult
       implements Serializable
{
  /**
   * A predefined result instance that indicates all processing completed
   * successfully.
   */
  public static final PreOperationPluginResult SUCCESS =
       new PreOperationPluginResult(false, true, false, false);



  /**
   * The serial version UID for this serializable class.
   */
  private static final long serialVersionUID = -255249379093850659L;



  // Indicates whether the client connection was terminated by the plugin.
  private final boolean connectionTerminated;

  // Indicates whether the server should continue processing other
  // pre-operation plugins.
  private final boolean continuePluginProcessing;

  // Indicates whether to send the response to the client immediately without
  // performing any of the core processing.
  private final boolean sendResponseImmediately;

  // Indicates whether the server should skip the core processing for the
  // associated operation.
  private final boolean skipCoreProcessing;



  /**
   * Creates a new pre-operation plugin result with the provided information.
   *
   * @param  connectionTerminated      Indicates whether the client connection
   *                                   was terminated by the plugin.
   * @param  continuePluginProcessing  Indicates whether to continue processing
   *                                   other pre-operation plugins for the
   *                                   operation.
   * @param  sendResponseImmediately   Indicates whether to send the response
   *                                   to the client immediately without
   *                                   performing any of the core processing.
   * @param  skipCoreProcessing        Indicates whether the server should skip
   *                                   the core processing for the operation.
   *                                   If this is {@code true} but
   *                                   {@code sendResponseImmediately} is
   *                                   {@code false}, then any post-operation
   *                                   plugins will still be invoked.
   */
  public PreOperationPluginResult(final boolean connectionTerminated,
                                  final boolean continuePluginProcessing,
                                  final boolean sendResponseImmediately,
                                  final boolean skipCoreProcessing)
  {
    this.connectionTerminated     = connectionTerminated;
    this.continuePluginProcessing = continuePluginProcessing;
    this.sendResponseImmediately  = sendResponseImmediately;
    this.skipCoreProcessing       = skipCoreProcessing;
  }



  /**
   * Indicates whether the client connection was terminated by the plugin.
   *
   * @return  {@code true} if the client connection was terminated by the
   *          plugin, or {@code false} if not.
   */
  public boolean connectionTerminated()
  {
    return connectionTerminated;
  }



  /**
   * Indicates whether to continue processing other pre-operation plugins for
   * the connection.
   *
   * @return  {@code true} if the server should continue processing other
   *          pre-operation plugins for the connection, or {@code false} if not.
   */
  public boolean continuePluginProcessing()
  {
    return continuePluginProcessing;
  }



  /**
   * Indicates whether the server should send a response to the client
   * immediately without performing any core processing.
   *
   * @return  {@code true} if the server should send a response to the client
   *          immediately without
   */
  public boolean sendResponseImmediately()
  {
    return sendResponseImmediately;
  }



  /**
   * Indicates whether the server should skip the core processing for the
   * operation.
   *
   * @return  {@code true} if the server should skip the core processing for the
   *          operation, or {@code false} if not.
   */
  public boolean skipCoreProcessing()
  {
    return skipCoreProcessing;
  }



  /**
   * Retrieves a string representation of this pre-operation plugin result.
   *
   * @return  A string representation of this pre-operation plugin result.
   */
  @Override()
  public String toString()
  {
    final StringBuilder buffer = new StringBuilder();

    buffer.append("PreOperationPluginResult(connectionTerminated=");
    buffer.append(connectionTerminated);
    buffer.append(", continuePluginProcessing=");
    buffer.append(continuePluginProcessing);
    buffer.append(", sendResponseImmediately=");
    buffer.append(sendResponseImmediately);
    buffer.append(", skipCoreProcessing=");
    buffer.append(skipCoreProcessing);
    buffer.append(')');

    return buffer.toString();
  }
}
