/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2012-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import java.util.List;

import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.directory.sdk.common.types.Entry;
import com.unboundid.ldap.sdk.Control;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface provides methods that may be used to construct SASL bind
 * result objects.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public interface SASLBindResultFactory
{
  /**
   * Creates a success SASL bind result in which the authentication and
   * authorization user identities are the same, and no diagnostic message,
   * controls, or server SASL credentials need to be returned.
   *
   * @param  authenticatedUserDN  The DN of the user that has been
   *                              authenticated.  It may be {@code null} if the
   *                              resulting authentication was anonymous.
   *
   * @return  The created success SASL bind result.
   */
  SuccessSASLBindResult createSuccessResult(final String authenticatedUserDN);



  /**
   * Creates a success SASL bind result with the provided information.
   *
   * @param  authenticatedUserDN    The DN of the user that has been
   *                                authenticated.  It may be empty or
   *                                {@code null} if the resulting authentication
   *                                was anonymous.
   * @param  authorizedUserDN       The DN of the user that should be used as
   *                                the authorization identity for subsequent
   *                                operations requested on the connection.  In
   *                                most cases, it should be the same as the
   *                                authenticated user DN, but it may be
   *                                different if an alternate authorization
   *                                identity was specified.  It may be empty or
   *                                {@code null} if the authorization identity
   *                                should be that of the anonymous user.
   * @param  diagnosticMessage      The diagnostic message that should be
   *                                included in the response to the client.  It
   *                                may be {@code null} if no diagnostic message
   *                                is needed.
   * @param  controls               The set of controls that should be included
   *                                in the response to the client.  It may be
   *                                {@code null} or empty if no response
   *                                controls are needed.
   * @param  serverSASLCredentials  The server SASL credentials that should be
   *                                included in the response to the client.  It
   *                                may be {@code null} if no server SASL
   *                                credentials are needed.
   *
   * @return  The created success SASL bind result.
   */
  SuccessSASLBindResult createSuccessResult(final String authenticatedUserDN,
                             final String authorizedUserDN,
                             final String diagnosticMessage,
                             final List<Control> controls,
                             final ASN1OctetString serverSASLCredentials);



  /**
   * Creates a success SASL bind result with the provided information.
   *
   * @param  authenticatedUserDN    The DN of the user that has been
   *                                authenticated.  It may be empty or
   *                                {@code null} if the resulting authentication
   *                                was anonymous.
   * @param  authorizedUserDN       The DN of the user that should be used as
   *                                the authorization identity for subsequent
   *                                operations requested on the connection.  In
   *                                most cases, it should be the same as the
   *                                authenticated user DN, but it may be
   *                                different if an alternate authorization
   *                                identity was specified.  It may be empty or
   *                                {@code null} if the authorization identity
   *                                should be that of the anonymous user.
   * @param  diagnosticMessage      The diagnostic message that should be
   *                                included in the response to the client.  It
   *                                may be {@code null} if no diagnostic message
   *                                is needed.
   * @param  controls               The set of controls that should be included
   *                                in the response to the client.  It may be
   *                                {@code null} or empty if no response
   *                                controls are needed.
   * @param  serverSASLCredentials  The server SASL credentials that should be
   *                                included in the response to the client.  It
   *                                may be {@code null} if no server SASL
   *                                credentials are needed.
   * @param  passwordUsed           The plaintext password that was used to
   *                                authenticate.  This may be {@code null} if
   *                                the associated SASL mechanism is not
   *                                password-based or if the plaintext password
   *                                is not available.
   *
   * @return  The created success SASL bind result.
   */
  SuccessSASLBindResult createSuccessResult(final String authenticatedUserDN,
                             final String authorizedUserDN,
                             final String diagnosticMessage,
                             final List<Control> controls,
                             final ASN1OctetString serverSASLCredentials,
                             final ASN1OctetString passwordUsed);



  /**
   * Creates a continuation SASL bind result (indicating that more processing
   * is required to complete the authentication) with the provided information.
   *
   * @param  diagnosticMessage      The diagnostic message that should be
   *                                included in the response to the client.  It
   *                                may be {@code null} if no diagnostic message
   *                                is needed.
   * @param  controls               The set of controls that should be included
   *                                in the response to the client.  It may be
   *                                {@code null} or empty if no response
   *                                controls are needed.
   * @param  serverSASLCredentials  The server SASL credentials that should be
   *                                included in the response to the client.  It
   *                                may be {@code null} if no server SASL
   *                                credentials are needed.
   *
   * @return  The created continuation SASL bind result.
   */
  ContinuationSASLBindResult createContinuationResult(
                                  final String diagnosticMessage,
                                  final List<Control> controls,
                                  final ASN1OctetString serverSASLCredentials);



  /**
   * Creates a failure SASL bind result with the provided information.
   *
   * @param  authenticationFailureReason  A message that explains the reason for
   *                                      the authentication failure.  This will
   *                                      be recorded in the server access log
   *                                      but not included in the response to
   *                                      return to the client.
   * @param  diagnosticMessage            The diagnostic message that should be
   *                                      included in the response to the
   *                                      client.  It may be {@code null} if no
   *                                      diagnostic message is needed.
   * @param  matchedDN                    The matched DN that should be included
   *                                      in the response to the client.  It may
   *                                      be {@code null} if no matched DN is
   *                                      needed.
   * @param  controls                     The set of controls that should be
   *                                      included in the response to the
   *                                      client.  It may be {@code null} or
   *                                      empty if no response controls are
   *                                      needed.
   * @param  serverSASLCredentials        The server SASL credentials that
   *                                      should be included in the response to
   *                                      the client.  It may be {@code null} if
   *                                      no server SASL credentials are needed.
   *
   * @return  The created failure SASL bind result.
   */
  FailureSASLBindResult createFailureResult(
                             final String authenticationFailureReason,
                             final String diagnosticMessage,
                             final String matchedDN,
                             final List<Control> controls,
                             final ASN1OctetString serverSASLCredentials);



  /**
   * Creates a failure SASL bind result with the provided information.
   *
   * @param  authenticationFailureReason        A message that explains the
   *                                            reason for the authentication
   *                                            failure.  This will be recorded
   *                                            in the server access log but not
   *                                            included in the response to
   *                                            return to the client.
   * @param  diagnosticMessage                  The diagnostic message that
   *                                            should be included in the
   *                                            response to the client.  It may
   *                                            be {@code null} if no diagnostic
   *                                            message is needed.
   * @param  matchedDN                          The matched DN that should be
   *                                            included in the response to the
   *                                            client.  It may be {@code null}
   *                                            if no matched DN is needed.
   * @param  controls                           The set of controls that should
   *                                            be included in the response to
   *                                            the client.  It may be
   *                                            {@code null} or empty if no
   *                                            response controls are needed.
   * @param  serverSASLCredentials              The server SASL credentials that
   *                                            should be included in the
   *                                            response to the client.  It may
   *                                            be {@code null} if no server
   *                                            SASL credentials are needed.
   * @param  unsuccessfullyAuthenticatedUserDN  The DN of the user that tried
   *                                            to authenticate but was unable
   *                                            to do so successfully, if
   *                                            applicable.
   *
   * @return  The created failure SASL bind result.
   */
  FailureSASLBindResult createFailureResult(
                             final String authenticationFailureReason,
                             final String diagnosticMessage,
                             final String matchedDN,
                             final List<Control> controls,
                             final ASN1OctetString serverSASLCredentials,
                             final String unsuccessfullyAuthenticatedUserDN);



  /**
   * Indicates whether the provided password is valid for the specified user.
   * Note that absolutely no password policy processing will be performed.  This
   * method merely determines whether the provided password is contained in the
   * specified user entry.
   *
   * @param  userDN    The DN of the user entry for which to make the
   *                   determination.  It must not be {@code null} or empty.
   * @param  password  The bytes comprising the non-encoded clear-text password
   *                   for which the determination is to be made.  It must not
   *                   be {@code null} or empty.
   *
   * @return  {@code true} if the given password is contained in the specified
   *          user entry, or {@code false} if not.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         make the determination.
   */
  boolean isUserPasswordValid(final String userDN,
                              final ASN1OctetString password)
          throws LDAPException;



  /**
   * Maps the provided username to a user entry using the identity mapper
   * associated with the SASL mechanism handler.
   *
   * @param  username  The username to be mapped to a user entry.
   *
   * @return  The entry for the user identified by the associated identity
   *          mapper.
   *
   * @throws  LDAPException  If no identity mapper is associated with the SASL
   *                         mechanism handler, or if the identity mapper cannot
   *                         be used to map the username to exactly one entry.
   */
  Entry mapUsernameToEntry(final String username)
        throws LDAPException;
}
