/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;



import java.io.Serializable;

import com.unboundid.util.NotMutable;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This class defines a structure which may be used to provide information about
 * the result of the processing performed by a startup plugin.
 */
@NotMutable()
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class StartupPluginResult
       implements Serializable
{
  /**
   * A predefined result instance that indicates all processing completed
   * successfully.
   */
  public static final StartupPluginResult SUCCESS =
       new StartupPluginResult(true, true, null);



  /**
   * The serial version UID for this serializable class.
   */
  private static final long serialVersionUID = -7383162206183763287L;



  // Indicates whether processing completed successfully.
  private final boolean completedSuccessfully;

  // Indicates whether server startup processing should continue.
  private final boolean continueStartup;

  // A message with information about any problem that occurred.
  private final String message;



  /**
   * Creates a new startup plugin result with the provided information.
   *
   * @param  completedSuccessfully  Indicates whether the plugin successfully
   *                                completed its processing.
   * @param  continueStartup        Indicates whether server startup processing
   *                                should continue.
   * @param  message                An optional message providing additional
   *                                information about the result of the plugin
   *                                processing.
   */
  public StartupPluginResult(final boolean completedSuccessfully,
                             final boolean continueStartup,
                             final CharSequence message)
  {
    this.completedSuccessfully = completedSuccessfully;
    this.continueStartup       = continueStartup;

    if (message == null)
    {
      this.message = null;
    }
    else
    {
      this.message = message.toString();
    }
  }



  /**
   * Indicates whether the plugin successfully completed its processing.
   *
   * @return  {@code true} if the plugin successfully completed its processing,
   *          or {@code false} if not.
   */
  public boolean completedSuccessfully()
  {
    return completedSuccessfully;
  }



  /**
   * Indicates whether the server startup process should continue.
   *
   * @return  {@code true} if the server startup process should continue, or
   *          {@code false} if not.
   */
  public boolean continueStartup()
  {
    return continueStartup;
  }



  /**
   * Retrieves a message with additional information about the processing
   * performed by this plugin, if available.
   *
   * @return  A message with additional information about the processing
   *          performed by this plugin, or {@code null} if none is available.
   */
  public String getMessage()
  {
    return message;
  }



  /**
   * Retrieves a string representation of this startup plugin result.
   *
   * @return  A string representation of this startup plugin result.
   */
  @Override()
  public String toString()
  {
    final StringBuilder buffer = new StringBuilder();

    buffer.append("StartupPluginResult(completedSuccessfully=");
    buffer.append(completedSuccessfully);
    buffer.append(", continueStartup=");
    buffer.append(continueStartup);

    if (message != null)
    {
      buffer.append(", message='");
      buffer.append(message);
      buffer.append('\'');
    }

    buffer.append(')');

    return buffer.toString();
  }
}
