/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.ds.types;


import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;


/**
 * This class implements an enumeration that may be used to control
 * the writability mode for the entire server or for a specific
 * backend. The writability mode may be "enabled", "disabled", or
 * "internal-only".
 */
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public enum WritabilityMode
{
  /**
   * Indicates that all write operations should be allowed.
   */
  ENABLED("enabled"),



  /**
   * Indicates that all write operations should be rejected.
   */
  DISABLED("disabled"),



  /**
   * Indicates that write operations from clients will be rejected,
   * but internal operations and updates through replication will
   * be allowed.
   */
  INTERNAL_ONLY("internal-only");



  // The human-readable name for this writability mode.
  private String modeName;



  /**
   * Creates a new {@code WritabilityMode} with the provided name.
   *
   * @param  modeName  The human-readable name for this {@code WritabilityMode}.
   */
  WritabilityMode(final String modeName)
  {
    this.modeName = modeName;
  }



  /**
   * Retrieves the {@code WritabilityMode} for the specified name.
   *
   * @param  modeName  The name of the {@code WritabilityMode} to retrieve.
   *
   * @return  The requested {@code WritabilityMode}, or <CODE>null</CODE> if
   *          the provided value is not the name of a valid mode.
   */
  public static WritabilityMode forName(final String modeName)
  {
    String lowerName = modeName.toLowerCase();
    if (lowerName.equals("enabled"))
    {
      return WritabilityMode.ENABLED;
    }
    else if (lowerName.equals("disabled"))
    {
      return WritabilityMode.DISABLED;
    }
    else if (lowerName.equals("internal-only") ||
             lowerName.equals("internal_only"))
    {
      return WritabilityMode.INTERNAL_ONLY;
    }
    else
    {
      return null;
    }
  }



  /**
   * Retrieves a string representation of this {@code WritabilityMode}.
   *
   * @return  A string representation of this {@code WritabilityMode}.
   */
  @Override
  public String toString()
  {
    return modeName;
  }
}
