/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.http.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.unboundid.directory.sdk.broker.internal.BrokerExtension;
import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.ds.internal.DirectoryServerExtension;
import com.unboundid.directory.sdk.http.config.HTTPOperationLoggerConfig;
import com.unboundid.directory.sdk.http.types.HTTPServerContext;
import com.unboundid.directory.sdk.metrics.internal.MetricsEngineExtension;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which
 * record information about interaction with HTTP clients.  HTTP operation
 * loggers may write information to files, but they may also write to other
 * locations, including databases, message, queues, e-mail messages, or other
 * targets.
 * <BR>
 * <H2>Configuring HTTP Operation Loggers</H2>
 * In order to configure an HTTP operation logger created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-log-publisher \
 *           --publisher-name "<I>{logger-name}</I>" \
 *           --type third-party-http-operation \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{logger-name}</I>" is the name to use for the HTTP operation logger
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends
 * {@code com.unboundid.directory.sdk.common.ds.HTTPOperationLogger}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the logger.  If multiple arguments should be provided to the
 * logger, then the "<CODE>--set extension-argument:<I>{name=value}</I></CODE>"
 * option should be provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.http.scripting.ScriptedHTTPOperationLogger
 */
@Extensible()
@DirectoryServerExtension()
@DirectoryProxyServerExtension(appliesToLocalContent=true,
     appliesToRemoteContent=true)
@MetricsEngineExtension()
@BrokerExtension()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class HTTPOperationLogger
       implements UnboundIDExtension,
                  Reconfigurable<HTTPOperationLoggerConfig>,
                  ExampleUsageProvider
{
  /**
   * The key that will be used to hold the request ID for the associated HTTP
   * request.  It will always be present in the state map when the
   * {@link #logRequest}  and {@link #logResponse} methods are invoked.  The
   * value associated with this key will be a {@code java.lang.Long} object.
   */
  public static final String STATE_KEY_REQUEST_ID =
       "___request_id___";



  /**
   * The key that will be used to hold the correlation ID for the associated
   * HTTP request.  It will be present in the state map when the
   * {@link #logRequest} and {@link #logResponse} methods are invoked, but only
   * if the HTTP Connection Handler is configured to use correlation IDs.  The
   * value associated with this key will be a {@code java.lang.String} object.
   */
  public static final String STATE_KEY_CORRELATION_ID =
      "___correlation_id___";



  /**
   * The key that will be used to hold the time the request was received, in
   * milliseconds since January 1, 1960, UTC, as reported by
   * {@code System.currentTimeMillis()}.  It will always be present in the state
   * map when the {@link #logRequest}  and {@link #logResponse} methods are
   * invoked.  The value associated with this key will be a
   * {@code java.lang.Long} object.
   */
  public static final String STATE_KEY_REQUEST_TIME_MILLIS =
       "___request_time_millis___";



  /**
   * The key that will be used to hold the time the request was received, in
   * nanoseconds, as reported by {@code System.nanoTime()}.  It will always be
   * present in the state map when the {@link #logRequest}  and
   * {@link #logResponse} methods are invoked.  The value associated with this
   * key will be a {@code java.lang.Long} object.
   */
  public static final String STATE_KEY_REQUEST_TIME_NANOS =
       "___request_time_nanos___";



  /**
   * The key that will be used to hold the response content length in the state
   * map.  It will always be present in the state map when the
   * {@link #logResponse} method is invoked.  The value associated with this key
   * will be a {@code java.lang.Long} object.
   */
  public static final String STATE_KEY_RESPONSE_CONTENT_LENGTH =
       "___response_content_length___";



  /**
   * The key that will be used to hold the operation processing time in
   * milliseconds.  It will always be present in the state map when the
   * {@link #logResponse} method is invoked.  The value associated with this key
   * will be a {@code java.lang.Long} object.
   */
  public static final String STATE_KEY_PROCESSING_TIME_MILLIS =
       "___processing_time_millis___";



  /**
   * The key that will be used to hold the operation processing time in
   * nanoseconds.  It will always be present in the state map when the
   * {@link #logResponse} method is invoked.  The value associated with this key
   * will be a {@code java.lang.Long} object.
   */
  public static final String STATE_KEY_PROCESSING_TIME_NANOS =
       "___processing_time_nanos___";



  /**
   * The key that will be used to hold the set of cookies included in the
   * response to the client.  It will always be present in the state map when
   * the {@link #logResponse} method is invoked.  The value associated with this
   * key will be a {@code java.util.List&lt;javax.servlet.http.Cookie&gt;}
   * object.
   */
  public static final String STATE_KEY_RESPONSE_COOKIES =
       "___response_cookies___";



  /**
   * Creates a new instance of this HTTP operation logger.  All HTTP operation
   * logger implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeHTTPOperationLogger} method.
   */
  public HTTPOperationLogger()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this HTTP operation logger.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this HTTP operation
   *                        logger.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this HTTP operation logger.
   *
   * @throws  LDAPException  If a problem occurs while initializing this HTTP
   *                         operation logger.
   */
  public void initializeHTTPOperationLogger(
                   final HTTPServerContext serverContext,
                   final HTTPOperationLoggerConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final HTTPOperationLoggerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final HTTPOperationLoggerConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this HTTP " +
                "operation logger is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this HTTP operation logger
   * is to be taken out of service.
   */
  public void finalizeHTTPOperationLogger()
  {
    // No implementation is required.
  }



  /**
   * Logs information about a servlet request that has been received from the
   * client.
   *
   * @param  request   An object with information about the request received
   *                   from the client.
   * @param  stateMap  An empty map which may be updated to hold state
   *                   information that can be used to correlate information
   *                   between the request and response.  The same map instance
   *                   will be passed to the {@link #logResponse} method.
   */
  public void logRequest(final HttpServletRequest request,
                         final Map<String,Object> stateMap)
  {
    // No processing performed by default.
  }



  /**
   * Logs information about a servlet response to be returned to the client.
   *
   * @param  request   An object with information about the request received
   *                   from the client.
   * @param  response  An object with information about the response to be
   *                   returned to the client.
   * @param  stateMap  A map containing state any information added while
   *                   processing the {@link #logRequest} method.
   */
  public void logResponse(final HttpServletRequest request,
                          final HttpServletResponse response,
                          final Map<String,Object> stateMap)
  {
    // No processing performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
