/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.proxy.config.LDAPHealthCheckConfig;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.proxy.types.BackendServer;
import com.unboundid.directory.sdk.proxy.types.CompletedProxyOperationContext;
import com.unboundid.directory.sdk.proxy.types.HealthCheckResult;
import com.unboundid.directory.sdk.proxy.types.ProxyServerContext;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which are
 * used to assess the health of backend servers accessed through the Directory
 * Proxy Server.  Each health check invocation generates a health check result,
 * which contains the following elements:
 * <UL>
 *   <LI>Health Check State -- This indicates the general health state for the
 *       backend server.  The state may be AVAILABLE (the server is completely
 *       suitable for use), DEGRADED (the server should be avoided if there are
 *       better servers but may be used if necessary), or UNAVAILABLE (the
 *       server should not be used at all).</LI>
 *   <LI>Health Check Score -- This provides an integer value between 10 (the
 *       best score) and 1 (the worst score) that may help rank servers with the
 *       same state.  Some load-balancing algorithms (e.g., the health-weighted
 *       algorithm) may use this to prefer servers with a higher score over
 *       those with a lower score.  Servers with a state of UNAVAILABLE will
 *       always have a score of zero.</LI>
 *   <LI>Messages -- This may optionally provide information about the reason
 *       for the state and/or score.  This is primarily useful for results
 *       indicating that the server is DEGRADED or UNAVAILABLE to provide
 *       information about the problem(s) preventing it from being considered
 *       AVAILABLE.</LI>
 * </UL>
 * <BR><BR>
 * LDAP health checks may be invoked in two ways.  They will be invoked on a
 * regular basis to periodically re-assess the health of each backend server,
 * but they may also be invoked after a failed operation in order to more
 * quickly detect a problem that should cause the server to be transitioned to a
 * less-available state.  The server will ensure that no more than one health
 * check is in progress at a time for a given server in order to avoid
 * overwhelming it with health checking, but it is still recommended that
 * health checks triggered as a result of a failed operation attempt to use the
 * operation result code to decide whether it may be necessary to actually
 * attempt to communicate with the server.
 * <BR><BR>
 * Further, it may also be useful to have more stringent criteria for promoting
 * the health of a server than for demoting it in order to avoid a ping-pong
 * effect that may occur if a server is hovering near the border between
 * AVAILABLE and DEGRADED or between DEGRADED and UNAVAILABLE.  For example, if
 * a health check attempts to perform an operation in the backend server and the
 * response time for that operation is taken into account when determining the
 * server health, it might be better to require a lower response time in order
 * to transition from DEGRADED to AVAILABLE than was originally required to
 * transition from AVAILABLE to DEGRADED.
 * <BR>
 * <H2>Configuring LDAP Health Checks</H2>
 * In order to configure an LDAP health check created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-ldap-health-check \
 *           --check-name "<I>{check-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{check-name}</I>" is the name to use for the LDAP health check
 * instance, "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends {@code com.unboundid.directory.sdk.proxy.api.LDAPHealthCheck},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the LDAP health check.  If multiple arguments should be provided
 * to the LDAP health check, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.proxy.scripting.ScriptedLDAPHealthCheck
 */
@Extensible()
@DirectoryProxyServerExtension(appliesToLocalContent=false,
     appliesToRemoteContent=true)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class LDAPHealthCheck
       implements UnboundIDExtension, Reconfigurable<LDAPHealthCheckConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this LDAP health check.  All LDAP health check
   * implementations must include a default constructor, but any initialization
   * should generally be done in the {@code initializeLDAPHealthCheck} method.
   */
  public LDAPHealthCheck()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this LDAP health check.
   *
   * @param  serverContext  A handle to the server context for the Directory
   *                        Proxy server in which this extension is running.
   * @param  config         The general configuration for this LDAP health
   *                        check.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this LDAP health check.
   *
   * @throws  LDAPException  If a problem occurs while initializing this LDAP
   *                         health check.
   */
  public void initializeLDAPHealthCheck(final ProxyServerContext serverContext,
                                        final LDAPHealthCheckConfig config,
                                        final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(final LDAPHealthCheckConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final LDAPHealthCheckConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this LDAP health " +
                "check is disabled and re-enabled or until the server is " +
                "restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this LDAP health check is
   * to be taken out of service.
   */
  public void finalizeLDAPHealthCheck()
  {
    // No implementation is required.
  }



  /**
   * Attempts to determine the health of the provided LDAP external server whose
   * last health check result indicated that the server had a state of
   * AVAILABLE.  This method may be periodically invoked for AVAILABLE servers
   * to determine whether their state has changed.
   *
   * @param  backendServer  A handle to the LDAP external server whose health is
   *                        to be assessed.
   * @param  connection     A connection that may be used to communicate with
   *                        the server in the course of performing the
   *                        evaluation.  The health check should not do anything
   *                        which may alter the state of this connection.
   *
   * @return  Information about the result of the health check.
   */
  public abstract HealthCheckResult checkAvailableServer(
                                         final BackendServer backendServer,
                                         final LDAPConnection connection);



  /**
   * Attempts to determine the health of the provided LDAP external server whose
   * last health check result indicated that the server had a state of DEGRADED.
   * This method may be periodically invoked for DEGRADED servers to determine
   * whether their state has changed.
   *
   * @param  backendServer  A handle to the LDAP external server whose health is
   *                        to be assessed.
   * @param  connection     A connection that may be used to communicate with
   *                        the server in the course of performing the
   *                        evaluation.  The health check should not do anything
   *                        which may alter the state of this connection.
   *
   * @return  Information about the result of the health check.
   */
  public abstract HealthCheckResult checkDegradedServer(
                                         final BackendServer backendServer,
                                         final LDAPConnection connection);



  /**
   * Attempts to determine the health of the provided LDAP external server whose
   * last health check result indicated that the server had a state of
   * UNAVAILABLE.  This method may be periodically invoked for UNAVAILABLE
   * servers to determine whether their state has changed.
   *
   * @param  backendServer  A handle to the LDAP external server whose health is
   *                        to be assessed.
   * @param  connection     A connection that may be used to communicate with
   *                        the server in the course of performing the
   *                        evaluation.  The health check should not do anything
   *                        which may alter the state of this connection.
   *
   * @return  Information about the result of the health check.
   */
  public abstract HealthCheckResult checkUnavailableServer(
                                         final BackendServer backendServer,
                                         final LDAPConnection connection);



  /**
   * Attempts to determine the health of the provided LDAP external server in
   * which an attempted operation did not complete successfully.
   *
   * @param  operationContext  A handle to the operation context for the
   *                           operation that failed.
   * @param  exception         The exception caught when attempting to process
   *                           the associated operation in the backend server.
   * @param  backendServer     A handle to the backend server in which the
   *                           operation was processed.
   *
   * @return  Information about the result of the health check.
   */
  public abstract HealthCheckResult checkFailedOperation(
                       final CompletedProxyOperationContext operationContext,
                       final LDAPException exception,
                       final BackendServer backendServer);



  /**
   * Retrieves a string representation of this LDAP health check.
   *
   * @return  A string representation of this LDAP health check.
   */
  @Override()
  public final String toString()
  {
    final StringBuilder buffer = new StringBuilder();
    toString(buffer);
    return buffer.toString();
  }



  /**
   * Appends a string representation of this LDAP health check to the provided
   * buffer.
   *
   * @param  buffer  The buffer to which the string representation should be
   *                 appended.
   */
  public abstract void toString(final StringBuilder buffer);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
