/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.AddRequest;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.proxy.config.PlacementAlgorithmConfig;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.proxy.types.BackendSet;
import com.unboundid.directory.sdk.proxy.types.ProxyServerContext;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which are
 * used to select the backend set in which a new entry should be added in an
 * entry-balanced environment.  The decision may be based on a wide range of
 * factors, including the contents of the entry to be added, the number of
 * entries in each of the backend servers, or other kinds of criteria.
 * <BR><BR>
 * Note that the placement algorithm will only be used for entries which are
 * immediate subordinates of the entry which is the balancing point.  Entries
 * which are more than one level below the balancing point will automatically
 * be added into the same backend set as their parent entry.
 * <BR>
 * <H2>Configuring Placement Algorithms</H2>
 * In order to configure a placement algorithm created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-placement-algorithm \
 *           --algorithm-name "<I>{algorithm-name}</I>" \
 *           --processor-name "<I>{processor-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{algorithm-name}</I>" is the to use for the placement algorithm
 * instance, "<I>{processor-name}</I>" is the name of the entry-balancing
 * request processor for which the placement algorithm will be used,
 * "<I>{class-name}</I>" is the fully-qualified name of the Java class
 * that extends
 * {@code com.unboundid.directory.sdk.proxy.api.PlacementAlgorithm}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the placement algorithm.  If multiple arguments should be provided
 * to the placement algorithm, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryProxyServerExtension(appliesToLocalContent=false,
     appliesToRemoteContent=true)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class PlacementAlgorithm
       implements UnboundIDExtension, Reconfigurable<PlacementAlgorithmConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this placement algorithm.  All placement
   * algorithm implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializePlacementAlgorithm} method.
   */
  public PlacementAlgorithm()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this placement algorithm.
   *
   * @param  serverContext    A handle to the server context for the Directory
   *                          Proxy server in which this extension is running.
   * @param  config           The general configuration for this placement
   *                          algorithm.
   * @param  parser           The argument parser which has been initialized
   *                          from the configuration for this placement
   *                          algorithm..
   * @param  balancingBaseDN  The balancing base DN for the associated
   *                          entry-balancing request processor.
   * @param  backendSets      The list of backend sets that will be used with
   *                          the entry-balancing request processor.
   *
   * @throws  LDAPException  If a problem occurs while initializing this LDAP
   *                         health check.
   */
  public void initializePlacementAlgorithm(
                   final ProxyServerContext serverContext,
                   final PlacementAlgorithmConfig config,
                   final ArgumentParser parser,
                   final String balancingBaseDN,
                   final List<BackendSet> backendSets)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final PlacementAlgorithmConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final PlacementAlgorithmConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this placement " +
                "algorithm is disabled and re-enabled or until the server " +
                "is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this LDAP health check is
   * to be taken out of service.
   */
  public void finalizePlacementAlgorithm()
  {
    // No implementation is required.
  }



  /**
   * Adapts to a change in the backend sets configured for use with the
   * associated entry-balancing request processor.
   *
   * @param  balancingBaseDN  The updated balancing base DN for the associated
   *                          entry-balancing request processor.
   * @param  backendSets      The updated list of backend sets for the
   *                          associated entry-balancing request processor.
   */
  public abstract void applyBalancingConfigurationChange(
                            final String balancingBaseDN,
                            final List<BackendSet> backendSets);



  /**
   * Determines the backend set that should be used to process the specified
   * add operation.
   *
   * @param  operationContext  The operation context for the add operation
   *                           to be processed.
   * @param  addRequest        The add request being processed.
   *
   * @return  The backend set in which the add should be processed, or
   *          {@code null} if there is no appropriate backend set.
   */
  public abstract BackendSet selectBackendSet(
                                  final OperationContext operationContext,
                                  final AddRequest addRequest);



  /**
   * Select a new backend set for an entry that has been modified or a child
   * entry that has been added below an existing entry in a backend set.
   *
   * @param  operationContext  The operation context for the operation that was
   *                           processed.
   * @param  backendSet        The backend set that currently holds the entry.
   *
   * @return  The backend set where the modified or added entry should be
   *          relocated, or {@code null} if the entry should not be relocated.
   */
  public BackendSet selectRebalancingBackendSet(
                                  final OperationContext operationContext,
                                  final BackendSet backendSet)
  {
    // The default implementation is not do any rebalancing.
    return null;
  }



  /**
   * Determines whether this placement algorithm implements the
   * {@link #selectRebalancingBackendSet} method to rebalance existing entries.
   *
   * @return  {@code true} if this placement algorithm supports re-balancing,
   *          or {@code false} if not.
   */
  public boolean supportsRebalancing()
  {
    // The default implementation is that re-balancing is not supported.
    return false;
  }



  /**
   * Retrieves a string representation of this placement algorithm.
   *
   * @return  A string representation of this placement algorithm.
   */
  @Override()
  public final String toString()
  {
    final StringBuilder buffer = new StringBuilder();
    toString(buffer);
    return buffer.toString();
  }



  /**
   * Appends a string representation of this placement algorithm to the provided
   * buffer.
   *
   * @param  buffer  The buffer to which the string representation should be
   *                 appended.
   */
  public abstract void toString(final StringBuilder buffer);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
