/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2014-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.common.operation.ExtendedRequest;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.directory.sdk.proxy.config.
            ProxiedExtendedOperationHandlerConfig;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.proxy.types.BackendSet;
import com.unboundid.directory.sdk.proxy.types.EntryBalancingRequestProcessor;
import com.unboundid.directory.sdk.proxy.types.ProxyingRequestProcessor;
import com.unboundid.directory.sdk.proxy.types.ProxyServerContext;
import com.unboundid.ldap.sdk.ExtendedResult;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ObjectPair;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which are
 * used to forward extended operations to one or more backend servers.  This
 * API makes it possible to select the backend set(s) to which the operation
 * should be forwarded, and to aggregate the responses into a single response to
 * return to the client.
 * <BR>
 * <H2>Configuring Proxied Extended Operation Handlers</H2>
 * In order to configure a proxied extended operation handler created using this
 * API, use a command like:
 * <PRE>
 *      dsconfig create-extended-operation-handler \
 *           --handler-name "<I>{handler-name}</I>" \
 *           --type third-party-proxied \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{handler-name}</I>" is the name to use for the proxied extended
 * operation handler instance, "<I>{class-name}</I>" is the fully-qualified name
 * of the Java class that extends {@code
 * com.unboundid.directory.sdk.proxy.api.ProxiedExtendedOperationHandler}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the proxied extended operation handler.  If multiple arguments
 * should be provided to the proxied extended operation handler, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryProxyServerExtension(appliesToLocalContent=false,
     appliesToRemoteContent=true)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ProxiedExtendedOperationHandler
       implements UnboundIDExtension,
                  Reconfigurable<ProxiedExtendedOperationHandlerConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this proxied extended operation handler.  All
   * proxied extended operation handler implementations must include a default
   * constructor, but any initialization should generally be done in the
   * {@code initializeProxiedExtendedOperationHandler} method.
   */
  public ProxiedExtendedOperationHandler()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  @Override()
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this proxied extended operation handler.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this extended
   *                        operation handler.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this extended operation
   *                        handler.
   *
   * @throws  LDAPException  If a problem occurs while initializing this
   *                         extended operation handler.
   */
  public void initializeProxiedExtendedOperationHandler(
                   final ProxyServerContext serverContext,
                   final ProxiedExtendedOperationHandlerConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public boolean isConfigurationAcceptable(
                      final ProxiedExtendedOperationHandlerConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public ResultCode applyConfiguration(
                         final ProxiedExtendedOperationHandlerConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this proxied " +
                "extended operation handler is disabled and re-enabled or " +
                "until the server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this proxied extended
   * operation handler is to be taken out of service.
   */
  public void finalizeProxiedExtendedOperationHandler()
  {
    // No implementation is required.
  }



  /**
   * Retrieves the name of the extended operation with the provided OID.
   *
   * @param  oid  The OID of the extended operation for which to retrieve the
   *              corresponding name.
   *
   * @return  The name of the extended operation with the specified OID, or
   *          {@code null} if the specified OID is not recognized by this
   *          proxied extended operation handler.
   */
  public abstract String getExtendedOperationName(final String oid);



  /**
   * Retrieves the OIDs of the extended operation types supported by this
   * proxied extended operation handler.
   *
   * @return  The OIDs of the extended operation types supported by this proxied
   *          extended operation handler.  It must not be {@code null} or
   *          empty, and the contents of the set returned must not change over
   *          the life of this proxied extended operation handler.
   */
  public abstract Set<String> getSupportedExtensions();



  /**
   * Retrieves the OIDs of any controls supported by this proxied extended
   * operation handler.
   *
   * @return  The OIDs of any controls supported by this proxied extended
   *          operation handler.  It may be {@code null} or empty if this
   *          proxied extended operation handler does not support any controls.
   */
  public Set<String> getSupportedControls()
  {
    return Collections.emptySet();
  }



  /**
   * Retrieves the OIDs of any features supported by this proxied extended
   * operation handler that should be advertised in the server root DSE.
   *
   * @return  The OIDs of any features supported by this proxied extended
   *          operation handler.  It may be {@code null} or empty if this
   *          proxied extended operation handler does not support any features.
   */
  public Set<String> getSupportedFeatures()
  {
    return Collections.emptySet();
  }



  /**
   * Selects the entry-balancing backend set(s) to which the provided extended
   * request should be forwarded.  This method will only be invoked for
   * operations in which the requester's client connection policy includes one
   * or more subtree views which reference an entry-balancing request processor.
   * <BR><BR>
   * This method returns two groups of backend sets, with the first representing
   * an initial guess (e.g., based on information obtained from the
   * entry-balancing global index), and the second representing a fallback if
   * the initial guess was found to be incorrect.
   * <BR><BR>
   * If it can be determined that no backend sets associated with the provided
   * entry-balancing request processor should be used to process the extended
   * operation, then the object returned may have both elements set to
   * {@code null} or empty sets.  If it is possible to definitively determine
   * the set of backend sets to which the operation should be forwarded and no
   * fallback option is required, then the first element of the returned object
   * should be populated with a non-empty set and the second element should be
   * {@code null} or empty.
   *
   * @param  operationContext  The operation context for the extended operation.
   * @param  request           The extended request to be processed.
   * @param  requestProcessor  The entry-balancing request processor in which
   *                           the extended operation should be processed.
   *
   * @return  The set of backend sets to which the request should be forwarded.
   *          It may be {@code null} (or it may be an {@code ObjectPair} with
   *          both elements empty or {@code null}) if the request should not be
   *          forwarded to any backend set for the entry-balancing request
   *          processor.
   *
   * @throws  LDAPException  If the request should not be forwarded to any of
   *                         the backend sets for the provided entry-balancing
   *                         request processor, and the result contained in the
   *                         exception should be used instead.
   */
  public abstract ObjectPair<Set<BackendSet>,Set<BackendSet>> selectBackendSets(
                       final OperationContext operationContext,
                       final ExtendedRequest request,
                       final EntryBalancingRequestProcessor requestProcessor)
         throws LDAPException;



  /**
   * Obtains the extended result that should be used as a result of processing
   * an operation in one or more entry-balanced backend sets, or throws an
   * exception to indicate that the request should instead be forwarded to the
   * fallback server set(s).
   * <BR><BR>
   * This method will only be invoked for cases in which the
   * {@link #selectBackendSets} method indicates that multiple backend sets
   * should be accessed in the course of processing an extended request.
   *
   * @param  operationContext   The operation context for the extended
   *                            operation.
   * @param  request            The extended request that was processed.
   * @param  requestProcessor   The entry-balancing request processor in which
   *                            the extended operation was processed.
   * @param  results            A list of the extended results obtained from
   *                            processing the operation in the selected set of
   *                            backend sets, with each result paired with
   *                            information about the backend set from which it
   *                            was obtained.
   * @param  fallbackAvailable  Indicates whether a fallback group of backend
   *                            sets is available and may be used as a second
   *                            attempt at processing the operation if the
   *                            result from the initial attempt is not
   *                            acceptable.
   *
   * @return  An extended result that represents the merged result from the
   *          provided list of results.  It must not be {@code null}.
   *
   * @throws  LDAPException  To indicate that the initial set of results was not
   *                         acceptable and that the operation should instead be
   *                         forwarded to the fallback group of backend sets.
   *                         If no fallback set of results is available, then an
   *                         extended result will be generated from the content
   *                         of this exception.
   */
  public abstract ExtendedResult mergeEntryBalancedResults(
              final OperationContext operationContext,
              final ExtendedRequest request,
              final EntryBalancingRequestProcessor requestProcessor,
              final List<ObjectPair<ExtendedResult,BackendSet>> results,
              final boolean fallbackAvailable)
         throws LDAPException;



  /**
   * Indicates whether the provided extended request should be forwarded to one
   * of the servers associated with the provided proxying request processor.
   * Note that this method will not be invoked for proxying request processors
   * associated with an entry-balancing request processor.
   *
   * @param  operationContext  The operation context for the extended operation.
   * @param  request           The extended request to be processed.
   * @param  requestProcessor  The proxying request processor for which to
   *                           make the determination.
   *
   * @return  {@code true} if the extended request should be forwarded to one of
   *          the servers associated with the proxying request processor, or
   *          {@code false} if not.
   *
   * @throws  LDAPException  If the request should not be forwarded to a
   *                         backend server associated with the proxying request
   *                         processor, but the result contained in the
   *                         exception should be used instead.
   */
  public abstract boolean shouldForward(final OperationContext operationContext,
                               final ExtendedRequest request,
                               final ProxyingRequestProcessor requestProcessor)
         throws LDAPException;



  /**
   * Creates the final extended result to return to the client from the provided
   * list of results obtained from the set of entry-balanced and/or proxying
   * request processors to which the request was forwarded.
   *
   * @param  operationContext   The operation context for the extended
   *                            operation.
   * @param  request            The extended request that was processed.
   * @param  results            The results from all request processors to which
   *                            the request was forwarded.  It may be empty if
   *                            the request was not forwarded to any backend
   *                            servers, in which case this method must
   *                            construct an appropriate result.  It may have
   *                            only a single element if the request was only
   *                            forwarded to one server, and in many cases it
   *                            may be desirable to simply use that result as
   *                            the final result.  It may also have multiple
   *                            elements if the request was forwarded to
   *                            multiple backend servers, in which case this
   *                            method must determine whether to return one of
   *                            them to the client, or to construct a new result
   *                            to return instead.
   *
   * @return  The final extended result to be returned to the client.
   */
  public abstract ExtendedResult createFinalResult(
                                      final OperationContext operationContext,
                                      final ExtendedRequest request,
                                      final List<ExtendedResult> results);



  /**
   * {@inheritDoc}
   */
  @Override()
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
