/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.proxy.config.ProxyTransformationConfig;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.proxy.internal.SearchResultProvider;
import com.unboundid.directory.sdk.proxy.types.ProxyOperationContext;
import com.unboundid.directory.sdk.proxy.types.ProxyServerContext;
import com.unboundid.ldap.sdk.AddRequest;
import com.unboundid.ldap.sdk.BindRequest;
import com.unboundid.ldap.sdk.BindResult;
import com.unboundid.ldap.sdk.CompareRequest;
import com.unboundid.ldap.sdk.DeleteRequest;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPResult;
import com.unboundid.ldap.sdk.ModifyRequest;
import com.unboundid.ldap.sdk.ModifyDNRequest;
import com.unboundid.ldap.sdk.ReadOnlyAddRequest;
import com.unboundid.ldap.sdk.ReadOnlyCompareRequest;
import com.unboundid.ldap.sdk.ReadOnlyDeleteRequest;
import com.unboundid.ldap.sdk.ReadOnlyModifyRequest;
import com.unboundid.ldap.sdk.ReadOnlyModifyDNRequest;
import com.unboundid.ldap.sdk.ReadOnlySearchRequest;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.SearchRequest;
import com.unboundid.ldap.sdk.SearchResult;
import com.unboundid.ldap.sdk.SearchResultEntry;
import com.unboundid.ldap.sdk.SearchResultReference;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which are
 * used to transform requests and/or responses as they pass through the
 * Directory Proxy Server.  They may be used to alter the contents of any add,
 * bind, compare, delete, modify, modify DN, or search request or result passing
 * through the Directory Proxy Server, or they may cause a result to be returned
 * to the client without it being forwarded to a backend server.  In addition,
 * transformations may be applied to search result entries and references to be
 * returned to the client, or those entries and references may be silently
 * dropped so that they are not returned.
 * <BR>
 * <H2>Configuring Proxy Transformations</H2>
 * In order to configure a proxy transformation created using this API, use a
 * command like:
 * <PRE>
 *      dsconfig create-proxy-transformation \
 *           --transformation-name "<I>{transformation-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{transformation-name}</I>" is the name to use for the proxy
 * transformation instance, "<I>{class-name}</I>" is the fully-qualified name of
 * the Java class that extends
 * {@code com.unboundid.directory.sdk.proxy.api.ProxyTransformation}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the proxy transformation.  If multiple arguments should be
 * provided to the proxy transformation, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.proxy.scripting.ScriptedProxyTransformation
 */
@Extensible()
@DirectoryProxyServerExtension(appliesToLocalContent=false,
     appliesToRemoteContent=true)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ProxyTransformation
       implements UnboundIDExtension, Reconfigurable<ProxyTransformationConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this proxy transformation.  All proxy
   * transformation implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeProxyTransformation} method.
   */
  public ProxyTransformation()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this proxy transformation.
   *
   * @param  serverContext  A handle to the server context for the Directory
   *                        Proxy server in which this extension is running.
   * @param  config         The general configuration for this proxy
   *                        transformation.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this proxy transformation.
   *
   * @throws  LDAPException  If a problem occurs while initializing this proxy
   *                         transformation.
   */
  public void initializeProxyTransformation(
                   final ProxyServerContext serverContext,
                   final ProxyTransformationConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final ProxyTransformationConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final ProxyTransformationConfig config,
                                       final ArgumentParser parser,
                                       final List<String> adminActionsRequired,
                                       final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this proxy " +
                "transformation is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this proxy transformation
   * is to be taken out of service.
   */
  public void finalizeProxyTransformation()
  {
    // No implementation is required.
  }



  /**
   * Applies any necessary transformation to the provided add request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  addRequest        The add request to be transformed.
   *
   * @return  A new add request which has been transformed as necessary, or the
   *          original request if no transformation is required or the provided
   *          add request has been updated in place.
   *
   * @throws  LDAPException  If a problem is encountered while processing the
   *                         transformation, or if the provided request should
   *                         not be forwarded to a backend server.
   */
  public AddRequest transformAddRequest(
                         final ProxyOperationContext operationContext,
                         final AddRequest addRequest)
         throws LDAPException
  {
    return addRequest;
  }



  /**
   * Applies any necessary transformation to the provided add result.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  addRequest        The add request that was processed.
   * @param  addResult         The add result to be transformed.
   *
   * @return  A new add result which has been transformed as necessary, or the
   *          original result if no transformation is required or the provided
   *          add result has been updated in place.
   */
  public LDAPResult transformAddResult(
                         final ProxyOperationContext operationContext,
                         final ReadOnlyAddRequest addRequest,
                         final LDAPResult addResult)
  {
    return addResult;
  }



  /**
   * Applies any necessary transformation to the provided bind request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  bindRequest       The bind request to be transformed.
   *
   * @return  A new bind request which has been transformed as necessary, or the
   *          original request if no transformation is required.
   *
   * @throws  LDAPException  If a problem is encountered while processing the
   *                         transformation, or if the provided request should
   *                         not be forwarded to a backend server.
   */
  public BindRequest transformBindRequest(
                         final ProxyOperationContext operationContext,
                         final BindRequest bindRequest)
         throws LDAPException
  {
    return bindRequest;
  }



  /**
   * Applies any necessary transformation to the provided bind request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  bindRequest       The bind request that was processed.
   * @param  bindResult        The bind result to be transformed.
   *
   * @return  A new bind result which has been transformed as necessary, or the
   *          original result if no transformation is required.
   */
  public BindResult transformBindResult(
                         final ProxyOperationContext operationContext,
                         final BindRequest bindRequest,
                         final BindResult bindResult)
  {
    return bindResult;
  }



  /**
   * Applies any necessary transformation to the provided compare request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  compareRequest    The compare request to be transformed.
   *
   * @return  A new compare request which has been transformed as necessary, or
   *          the original request if no transformation is required or the
   *          provided compare request has been updated in place.
   *
   * @throws  LDAPException  If a problem is encountered while processing the
   *                         transformation, or if the provided request should
   *                         not be forwarded to a backend server.
   */
  public CompareRequest transformCompareRequest(
                             final ProxyOperationContext operationContext,
                             final CompareRequest compareRequest)
         throws LDAPException
  {
    return compareRequest;
  }



  /**
   * Applies any necessary transformation to the provided compare result.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  compareRequest    The compare request that was processed.
   * @param  compareResult     The compare result to be transformed.
   *
   * @return  A new compare result which has been transformed as necessary, or
   *          the original result if no transformation is required or the
   *          provided compare result has been updated in place.
   */
  public LDAPResult transformCompareResult(
                         final ProxyOperationContext operationContext,
                         final ReadOnlyCompareRequest compareRequest,
                         final LDAPResult compareResult)
  {
    return compareResult;
  }



  /**
   * Applies any necessary transformation to the provided delete request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  deleteRequest     The delete request to be transformed.
   *
   * @return  A new delete request which has been transformed as necessary, or
   *          the original request if no transformation is required or the
   *          provided delete request has been updated in place.
   *
   * @throws  LDAPException  If a problem is encountered while processing the
   *                         transformation, or if the provided request should
   *                         not be forwarded to a backend server.
   */
  public DeleteRequest transformDeleteRequest(
                            final ProxyOperationContext operationContext,
                            final DeleteRequest deleteRequest)
         throws LDAPException
  {
    return deleteRequest;
  }



  /**
   * Applies any necessary transformation to the provided delete result.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  deleteRequest     The delete request that was processed.
   * @param  deleteResult      The delete result to be transformed.
   *
   * @return  A new delete result which has been transformed as necessary, or
   *          the original result if no transformation is required or the
   *          provided delete result has been updated in place.
   */
  public LDAPResult transformDeleteResult(
                         final ProxyOperationContext operationContext,
                         final ReadOnlyDeleteRequest deleteRequest,
                         final LDAPResult deleteResult)
  {
    return deleteResult;
  }



  /**
   * Applies any necessary transformation to the provided modify request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  modifyRequest     The modify request to be transformed.
   *
   * @return  A new modify request which has been transformed as necessary, or
   *          the original request if no transformation is required or the
   *          provided modify request has been updated in place.
   *
   * @throws  LDAPException  If a problem is encountered while processing the
   *                         transformation, or if the provided request should
   *                         not be forwarded to a backend server.
   */
  public ModifyRequest transformModifyRequest(
                            final ProxyOperationContext operationContext,
                            final ModifyRequest modifyRequest)
         throws LDAPException
  {
    return modifyRequest;
  }



  /**
   * Applies any necessary transformation to the provided modify result.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  modifyRequest     The modify request that was processed.
   * @param  modifyResult      The modify result to be transformed.
   *
   * @return  A new modify result which has been transformed as necessary, or
   *          the original result if no transformation is required or the
   *          provided modify result has been updated in place.
   */
  public LDAPResult transformModifyResult(
                         final ProxyOperationContext operationContext,
                         final ReadOnlyModifyRequest modifyRequest,
                         final LDAPResult modifyResult)
  {
    return modifyResult;
  }



  /**
   * Applies any necessary transformation to the provided modify DN request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  modifyDNRequest   The modify DN request to be transformed.
   *
   * @return  A new modify DN request which has been transformed as necessary,
   *          or the original request if no transformation is required or the
   *          provided modify DN request has been updated in place.
   *
   * @throws  LDAPException  If a problem is encountered while processing the
   *                         transformation, or if the provided request should
   *                         not be forwarded to a backend server.
   */
  public ModifyDNRequest transformModifyDNRequest(
                              final ProxyOperationContext operationContext,
                              final ModifyDNRequest modifyDNRequest)
         throws LDAPException
  {
    return modifyDNRequest;
  }



  /**
   * Applies any necessary transformation to the provided modify DN result.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  modifyDNRequest   The modify DN request that was processed.
   * @param  modifyDNResult    The modify DN result to be transformed.
   *
   * @return  A new modify DN result which has been transformed as necessary, or
   *          the original result if no transformation is required or the
   *          provided modify DN result has been updated in place.
   */
  public LDAPResult transformModifyDNResult(
                         final ProxyOperationContext operationContext,
                         final ReadOnlyModifyDNRequest modifyDNRequest,
                         final LDAPResult modifyDNResult)
  {
    return modifyDNResult;
  }



  /**
   * Applies any necessary transformation to the provided search request.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  searchRequest     The search request to be transformed.
   *
   * @return  A new search request which has been transformed as necessary, or
   *          the original request if no transformation is required or the
   *          provided search request has been updated in place.
   *
   * @throws  LDAPException  If a problem is encountered while processing the
   *                         transformation, or if the provided request should
   *                         not be forwarded to a backend server.
   */
  public SearchRequest transformSearchRequest(
                            final ProxyOperationContext operationContext,
                            final SearchRequest searchRequest)
         throws LDAPException
  {
    return searchRequest;
  }



  /**
   * Applies any necessary transformation to the provided search result.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  searchRequest     The search request that was processed.
   * @param  searchResult      The search result to be transformed.
   * @param  resultProvider    The search result provider which may be used to
   *                           send additional search result entries and/or
   *                           references to the client.
   *
   * @return  A new search result which has been transformed as necessary, or
   *          the original result if no transformation is required or the
   *          provided search result has been updated in place.
   */
  public SearchResult transformSearchResult(
                           final ProxyOperationContext operationContext,
                           final ReadOnlySearchRequest searchRequest,
                           final SearchResult searchResult,
                           final SearchResultProvider resultProvider)
  {
    return searchResult;
  }



  /**
   * Applies any necessary transformation to the provided search result entry.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  searchRequest     The search request that is being processed.
   * @param  searchEntry       The search result entry to be transformed.
   *
   * @return  A new search result entry which has been transformed as necessary,
   *          the original search result entry if no transformation is required,
   *          or {@code null} if the entry should not be returned to the client.
   */
  public SearchResultEntry transformSearchResultEntry(
                                final ProxyOperationContext operationContext,
                                final ReadOnlySearchRequest searchRequest,
                                final SearchResultEntry searchEntry)
  {
    return searchEntry;
  }



  /**
   * Applies any necessary transformation to the provided search result
   * reference.
   *
   * @param  operationContext  Information about the operation being processed
   *                           in the Directory Proxy Server.
   * @param  searchRequest     The search request that is being processed.
   * @param  searchReference   The search result reference to be transformed.
   *
   * @return  A new search result reference which has been transformed as
   *          necessary, the original search result reference if no
   *          transformation is required, or {@code null} if the reference
   *          should not be returned to the client.
   */
  public SearchResultReference transformSearchResultReference(
              final ProxyOperationContext operationContext,
              final ReadOnlySearchRequest searchRequest,
              final SearchResultReference searchReference)
  {
    return searchReference;
  }



  /**
   * Retrieves a string representation of this proxy transformation.
   *
   * @return  A string representation of this proxy transformation.
   */
  @Override()
  public final String toString()
  {
    final StringBuilder buffer = new StringBuilder();
    toString(buffer);
    return buffer.toString();
  }



  /**
   * Appends a string representation of this proxy transformation to the
   * provided buffer.
   *
   * @param  buffer  The buffer to which the string representation should be
   *                 appended.
   */
  public abstract void toString(final StringBuilder buffer);



  /**
   * {@inheritDoc}
   */
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
