/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2014-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.api;



import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.proxy.config.ServerAffinityProviderConfig;
import com.unboundid.directory.sdk.proxy.internal.DirectoryProxyServerExtension;
import com.unboundid.directory.sdk.proxy.types.BackendServer;
import com.unboundid.directory.sdk.proxy.types.ProxyOperationContext;
import com.unboundid.directory.sdk.proxy.types.ProxyServerContext;
import com.unboundid.directory.sdk.proxy.types.ServerAffinity;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by extensions which are
 * used to influence which backend server a client should use in a load-balanced
 * set.  The primary purpose is to attempt to ensure that related requests are
 * sent to the same backend server in an attempt to eliminate problems that may
 * result from replication propagation delay (e.g., because of a read
 * immediately after a write), to ensure that repeated accesses to the same data
 * benefit from having the data in-cache, etc.  A server affinity provider is
 * called at two points in operation processing:
 * <UL>
 *   <LI>Before using the load-balancing algorithm to select a backend server.
 *       If the server affinity provider indicates that there is already an
 *       affinity defined for the operation, and the affinity is for a server
 *       that has a health check state of AVAILABLE, then the server selected by
 *       affinity will be used instead selecting a server with the
 *       load-balancing algorithm.  If the server affinity provider does not
 *       specify which server to use, or if the server selected by the affinity
 *       provider does not have a health check state of AVAILABLE, then the
 *       load-balancing algorithm will be used to select the server.</LI>
 *   <LI>After an operation has been processed.  This may be used to set or
 *       update state information that may be used by the next request in a
 *       "related" set (where "related" is defined according to whatever logic
 *       the affinity provider provides).</LI>
 * </UL>
 * <BR><BR>
 * <H2>Configuring Server Affinity Providers</H2>
 * In order to configure a server affinity provider created using this API, use
 * a command like:
 * <PRE>
 *      dsconfig create-server-affinity-provider \
 *           --provider-name "<I>{provider-name}</I>" \
 *           --type third-party \
 *           --set enabled:true \
 *           --set "extension-class:<I>{class-name}</I>" \
 *           --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{provider-name}</I>" is the name to use for the server affinity
 * provider instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Java class that extends
 * {@code com.unboundid.directory.sdk.proxy.api.ServerAffinityProvider}, and
 * "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the server affinity provider.  If multiple arguments should be
 * provided to the server affinity provider, then the
 * "<CODE>--set extension-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 */
@Extensible()
@DirectoryProxyServerExtension(appliesToLocalContent=false,
     appliesToRemoteContent=true)
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ServerAffinityProvider
       implements UnboundIDExtension,
                  Reconfigurable<ServerAffinityProviderConfig>,
                  ExampleUsageProvider
{
  /**
   * Creates a new instance of this server affinity provider.  All server
   * affinity provider implementations must include a default constructor, but
   * any initialization should generally be done in the
   * {@code initializeServerAffinityProvider} method.
   */
  public ServerAffinityProvider()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String getExtensionName();



  /**
   * {@inheritDoc}
   */
  @Override()
  public abstract String[] getExtensionDescription();



  /**
   * {@inheritDoc}
   */
  @Override()
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this server affinity provider.
   *
   * @param  serverContext  A handle to the server context for the Directory
   *                        Proxy server in which this extension is running.
   * @param  config         The general configuration for this server affinity
   *                        provider.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this server affinity provider.
   *
   * @throws  LDAPException  If a problem occurs while initializing this server
   *                         affinity provider.
   */
  public void initializeServerAffinityProvider(
                   final ProxyServerContext serverContext,
                   final ServerAffinityProviderConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public boolean isConfigurationAcceptable(
                      final ServerAffinityProviderConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  @Override()
  public ResultCode applyConfiguration(
                         final ServerAffinityProviderConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.  If there are any
    // arguments, then add an admin action message indicating that the extension
    // needs to be restarted for any changes to take effect.
    if (! parser.getNamedArguments().isEmpty())
    {
      adminActionsRequired.add(
           "No configuration change has actually been applied.  The new " +
                "configuration will not take effect until this server " +
                "affinity provider is disabled and re-enabled or until the " +
                "server is restarted.");
    }

    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup which may be necessary when this server affinity
   * provider is to be taken out of service.
   */
  public void finalizeServerAffinityProvider()
  {
    // No implementation is required.
  }



  /**
   * Clears all affinity data associated with the provided list of
   * load-balancing algorithms.
   *
   * @param  lbaDN           The config entry DN of the load-balancing algorithm
   *                         for which to clear the affinity data.  If this is
   *                         {@code null}, then all affinity data for all
   *                         load-balancing algorithms should be cleared.
   * @param  backendServers  The set of backend servers that are associated with
   *                         the specified load-balancing algorithm.
   */
  public abstract void clearAffinityData(final DN lbaDN,
                            final Map<DN,BackendServer> backendServers);



  /**
   * Indicates which backend server should be used for the provided operation.
   * It is generally recommended that this method only return a server if the
   * operation matches an affinity that has already been established (via a
   * previous call to the {@code updateAffinity} method).  If no affinity has
   * been set, then it is recommended that this method return {@code null} to
   * allow the load-balancing algorithm to select an appropriate server instead.
   *
   * @param  lbaDN           The config entry DN of the load-balancing algorithm
   *                         for which to make the determination.
   * @param  backendServers  The set of backend servers from which the selection
   *                         may be made, indexed by the DN of the configuration
   *                         entry for each server.  It will not be
   *                         {@code null}.
   * @param  operation       The operation to be processed.  It will not be
   *                         {@code null}.
   *
   * @return  The backend server for which an affinity is already established,
   *          or {@code null} if the operation does not match any affinity that
   *          has already been established and the appropriate backend server
   *          should be selected by the load-balancing algorithm.
   */
  public abstract ServerAffinity selectServer(final DN lbaDN,
                       final Map<DN,BackendServer> backendServers,
                       final ProxyOperationContext operation);



  /**
   * Specifies the backend server that was used to process the provided
   * operation, which allows this affinity provider to establish or update any
   * necessary state information that could be used to select the same server
   * for "related" operations that may be processed in the future.
   *
   * @param  operation      The operation that was processed.
   * @param  lbaDN          The config entry DN of the load-balancing algorithm
   *                        with which the backend server is associated.
   * @param  backendServer  The backend server that was used to process the
   *                        operation.
   */
  public abstract void updateAffinity(
                            final ProxyOperationContext operation,
                            final DN lbaDN, final BackendServer backendServer);



  /**
   * Retrieves a string representation of this server affinity provider.
   *
   * @return  A string representation of this server affinity provider.
   */
  @Override()
  public final String toString()
  {
    final StringBuilder buffer = new StringBuilder();
    toString(buffer);
    return buffer.toString();
  }



  /**
   * Appends a string representation of this server affinity provider to the
   * provided buffer.
   *
   * @param  buffer  The buffer to which the string representation should be
   *                 appended.
   */
  public abstract void toString(final StringBuilder buffer);



  /**
   * {@inheritDoc}
   */
  @Override()
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
