/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.types;



import com.unboundid.directory.sdk.common.types.OperationType;
import com.unboundid.ldap.sdk.AddRequest;
import com.unboundid.ldap.sdk.CompareRequest;
import com.unboundid.ldap.sdk.CompareResult;
import com.unboundid.ldap.sdk.DeleteRequest;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPResult;
import com.unboundid.ldap.sdk.ModifyRequest;
import com.unboundid.ldap.sdk.ModifyDNRequest;
import com.unboundid.ldap.sdk.SearchRequest;
import com.unboundid.ldap.sdk.SearchResult;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to interact with an
 * LDAP external server which acts as a backend server accessed through the
 * Directory Proxy Server.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface BackendServer
{
  /**
   * Retrieves the DN of the configuration entry in which this LDAP external
   * server is defined.
   *
   * @return  The DN of the configuration entry in which this LDAP external
   *          server is defined.
   */
  String getConfigEntryDN();



  /**
   * Retrieves the DN of the configuration entry in which this LDAP external
   * server is defined.
   *
   * @return  The DN of the configuration entry in which this LDAP external
   *          server is defined.
   */
  DN getParsedConfigEntryDN();



  /**
   * Creates a new connection to the associated directory server which is
   * authenticated and/or secured in accordance with the LDAP external server
   * configuration.  The caller will be responsible for closing the connection
   * when it is no longer needed.
   *
   * @param  options  The set of options to use for the connection.  It may be
   *                  {@code null} if a default set of connection options should
   *                  be used.
   * @param  purpose  A message with information about the purpose for this
   *                  connection.  This message will be included in the
   *                  administrative operation control sent to the server (if
   *                  the server supports that control) for any operations
   *                  processed during the course of preparing the connection.
   *
   * @return  The newly-created connection.
   *
   * @throws  LDAPException  If a problem is encountered while attempting to
   *                         create the connection.
   */
  LDAPConnection createNewConnection(final LDAPConnectionOptions options,
                                     final String purpose)
                 throws LDAPException;



  /**
   * Retrieves the location assigned to this LDAP external server, if any.
   *
   * @return  The location assigned to this LDAP external server, or
   *          {@code null} if no location has been assigned.
   */
  Location getLocation();



  /**
   * Retrieves the address for this LDAP external server.
   *
   * @return  The address for this LDAP external server.
   */
  String getServerAddress();



  /**
   * Retrieves the port for this LDAP external server.
   *
   * @return  The port for this LDAP external server.
   */
  int getServerPort();



  /**
   * Indicates whether this LDAP external server is in the same location as the
   * Directory Proxy Server that is referencing it.
   *
   * @return  {@code true} if this LDAP external server is in the same location
   *          as the Directory Proxy Server that is referencing it, or
   *          {@code false} if it is in a different location or no location has
   *          been assigned.
   */
  boolean isLocal();



  /**
   * Retrieves the number of operations which are currently being processed
   * against the LDAP external server through the local Directory Proxy Server
   * instance.  Note that this value will not include operations processed
   * against the server from other Directory Proxy Server instances or any other
   * kind of client, nor will it include any operations which may be in progress
   * on connections created from the {@link #createNewConnection} method.
   *
   * @return  The number of operations which are currently being processed
   *          against the LDAP external server through the local Directory Proxy
   *          Server instance.
   */
  int getActiveOperations();



  /**
   * Retrieves the health check result currently assigned to this LDAP external
   * server.
   *
   * @return  The health check result currently assigned to this LDAP external
   *          server.
   */
  HealthCheckResult getHealthCheckResult();



  /**
   * Indicates whether this LDAP external server currently has a health check
   * state of AVAILABLE.
   *
   * @return  {@code true} if this LDAP external server currently has a health
   *          check state of AVAILABLE, or {@code false} if not.
   */
  boolean isAvailable();



  /**
   * Indicates whether this LDAP external server may be used to process
   * operations of the specified type.
   *
   * @param  t  The operation type for which to make the determination.
   *
   * @return  {@code true} if this LDAP external server may be used to process
   *          operations of the specified type, or {@code false} if not.
   */
  boolean supportsOperation(final OperationType t);



  /**
   * Sends the provided add request to the backend server without altering the
   * request or result in any way.  It will use a pooled connection
   * authenticated as the proxy user, and the provided request must not alter
   * the state of the connection in any way.
   *
   * @param  addRequest  The add request to be processed.
   *
   * @return  The result of processing the add operation.
   *
   * @throws  LDAPException  If an error occurred while processing the add
   *                         request.
   */
  LDAPResult addUnaltered(final AddRequest addRequest)
             throws LDAPException;



  /**
   * Sends the provided compare request to the backend server without altering
   * the request or result in any way.  It will use a pooled connection
   * authenticated as the proxy user, and the provided request must not alter
   * the state of the connection in any way.
   *
   * @param  compareRequest  The compare request to be processed.
   *
   * @return  The result of processing the compare operation.
   *
   * @throws  LDAPException  If an error occurred while processing the compare
   *                         request.
   */
  CompareResult compareUnaltered(final CompareRequest compareRequest)
                throws LDAPException;



  /**
   * Sends the provided delete request to the backend server without altering
   * the request or result in any way.  It will use a pooled connection
   * authenticated as the proxy user, and the provided request must not alter
   * the state of the connection in any way.
   *
   * @param  deleteRequest  The delete request to be processed.
   *
   * @return  The result of processing the delete operation.
   *
   * @throws  LDAPException  If an error occurred while processing the delete
   *                         request.
   */
  LDAPResult deleteUnaltered(final DeleteRequest deleteRequest)
             throws LDAPException;



  /**
   * Sends the provided modify request to the backend server without altering
   * the request or result in any way.  It will use a pooled connection
   * authenticated as the proxy user, and the provided request must not alter
   * the state of the connection in any way.
   *
   * @param  modifyRequest  The modify request to be processed.
   *
   * @return  The result of processing the modify operation.
   *
   * @throws  LDAPException  If an error occurred while processing the modify
   *                         request.
   */
  LDAPResult modifyUnaltered(final ModifyRequest modifyRequest)
             throws LDAPException;



  /**
   * Sends the provided modify DN request to the backend server without altering
   * the request or result in any way.  It will use a pooled connection
   * authenticated as the proxy user, and the provided request must not alter
   * the state of the connection in any way.
   *
   * @param  modifyDNRequest  The modify DN request to be processed.
   *
   * @return  The result of processing the modify DN operation.
   *
   * @throws  LDAPException  If an error occurred while processing the modify DN
   *                         request.
   */
  LDAPResult modifyDNUnaltered(final ModifyDNRequest modifyDNRequest)
             throws LDAPException;



  /**
   * Sends the provided search request to the backend server without altering
   * the request or result in any way.  It will use a pooled connection
   * authenticated as the proxy user, and the provided request must not alter
   * the state of the connection in any way.
   *
   * @param  searchRequest  The search request to be processed.
   *
   * @return  The result of processing the search operation.
   *
   * @throws  LDAPException  If an error occurred while processing the search
   *                         request.
   */
  SearchResult searchUnaltered(final SearchRequest searchRequest)
               throws LDAPException;
}
