/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.types;



import java.util.List;

import com.unboundid.directory.sdk.common.types.ClientContext;
import com.unboundid.directory.sdk.common.types.OperationContext;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.SearchResultEntry;
import com.unboundid.ldap.sdk.SearchScope;
import com.unboundid.util.NotExtensible;
import com.unboundid.util.ObjectPair;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to obtain
 * information about and interact with an entry-balancing request processor and
 * its component backend sets and servers.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface EntryBalancingRequestProcessor extends RequestProcessor
{
  /**
   * Retrieves the base DN for the entry-balancing request processor.
   *
   * @return  The base DN for the entry-balancing request processor.
   */
  DN getRequestProcessorBaseDN();



  /**
   * Retrieves the DN of the balancing point for the entry-balancing request
   * processor.
   *
   * @return  The DN of the balancing point for the entry-balancing request
   *          processor.
   */
  DN getBalancingPointBaseDN();



  /**
   * Retrieves a list of the backend sets for this entry-balancing request
   * processor.
   *
   * @return  A list of the backend sets for this entry-balancing request
   *          processor.
   */
  List<BackendSet> getBackendSets();



  /**
   * Retrieves the backend set in which the entry-balancing global index
   * believes the entry with the specified DN may exist.
   *
   * @param  dn  The DN of the entry for which to retrieve the global index
   *             hint.
   *
   * @return  The backend set in which the entry-balancing global index believes
   *          the entry with the specified DN may exist, or {@code null} if the
   *          global index does not have any information that may help locate
   *          the target entry, or if the provided DN is not below the balancing
   *          point.
   */
  BackendSet getGlobalIndexHint(final DN dn);



  /**
   * Retrieves the backend set in which the entry-balancing global index
   * believes the entry with the specified attribute value may exist.  The value
   * is expected to be unique across the entire data set.
   *
   * @param  attributeType  The name or OID of the attribute type of the
   *                        attribute for which to retrieve the global index
   *                        hint.
   * @param  value          The attribute value for which to retrieve the global
   *                        index hint.
   *
   * @return  The backend set in which the entry-balancing global index believes
   *          the entry with the specified attribute value may exist, or
   *          {@code null} if the global index does not have any information
   *          that may help locate the target entry.
   */
  BackendSet getGlobalIndexHint(final String attributeType, final String value);



  /**
   * Retrieves the backend set in which the entry-balancing global index
   * believes the entry with the specified attribute value may exist.  The value
   * is expected to be unique across the entire data set.
   *
   * @param  attributeType  The name or OID of the attribute type of the
   *                        attribute for which to retrieve the global index
   *                        hint.
   * @param  value          The attribute value for which to retrieve the global
   *                        index hint.
   *
   * @return  The backend set in which the entry-balancing global index believes
   *          the entry with the specified attribute value may exist, or
   *          {@code null} if the global index does not have any information
   *          that may help locate the target entry.
   */
  BackendSet getGlobalIndexHint(final String attributeType, final byte[] value);



  /**
   * Retrieves the specified entry from this entry-balancing request processor.
   * If a broadcast is needed to find the entry, it will be done in parallel.
   *
   * @param  op  Information about the operation currently being processed.
   *             This must not be {@code null}.
   * @param  dn  The DN of the entry to retrieve.  It must not be {@code null}.
   *             If it is at or above the balancing point base DN, then the
   *             backend set from which the entry will be retrieved is
   *             undefined.
   *
   * @return  An object that pairs the requested entry with information about
   *          the backend set from which it was retrieved, or {@code null} if
   *          the specified entry was not found in any backend set.
   *
   * @throws  LDAPException  If a problem is encountered while making the
   *                         determination.
   */
  ObjectPair<SearchResultEntry,BackendSet> getEntry(final OperationContext op,
                                                    final String dn)
       throws LDAPException;



  /**
   * Performs a search for all entries in this entry-balancing request processor
   * that match the provided criteria.  The search will not be allowed to return
   * more than 100 entries.
   *
   * @param  clientContext  The client context to use when performing the
   *                        search.  If this is {@code null}, then an internal
   *                        root user will be used to retrieve the entry.  If it
   *                        is non-{@code null}, then its authorization identity
   *                        will be used.
   * @param  baseDN         The base DN to use for the search.
   * @param  scope          The scope to use for the search.
   * @param  filter         The filter to use for the search.
   * @param  attributes     The set of attributes to request to be included in
   *                        the entries that are returned.  If this is
   *                        {@code null} or empty, then all user attributes will
   *                        be requested.
   *
   * @return  A list containing each matching entry paired with the backend set
   *          from which it was retrieved.
   *
   * @throws  LDAPException  If a problem is encountered while performing the
   *                         search.
   */
  List<ObjectPair<SearchResultEntry,BackendSet>> search(
           final ClientContext clientContext, final String baseDN,
           final SearchScope scope, final Filter filter,
           final String... attributes)
       throws LDAPException;
}
