/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.types;



import java.io.Serializable;
import java.util.Comparator;
import java.util.List;

import com.unboundid.util.NotExtensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This interface defines a set of methods that may be used to assess the health
 * of an LDAP external server.  A health check result consists of both a state
 * (AVAILABLE, DEGRADED, or UNAVAILABLE) and a numeric score that may be used to
 * help differentiate servers with the same health check state.
 */
@NotExtensible()
@ThreadSafety(level=ThreadSafetyLevel.INTERFACE_NOT_THREADSAFE)
public interface HealthCheckResult
       extends Comparable<HealthCheckResult>, Comparator<HealthCheckResult>,
               Serializable
{
  /**
   * Retrieves the health check state for the associated server.
   *
   * @return  The health check state for the associated server.
   */
  HealthCheckState getState();



  /**
   * Retrieves the numeric score for the associated server, which may be used to
   * help differentiate the server from other servers with the same state.  The
   * value returned will be an integer between 10 (the most healthy) and 1 (the
   * least healthy) for a state of AVAILABLE or DEGRADED, or zero for a state
   * of UNAVAILABLE.
   *
   * @return  The numeric score for the associated server.
   */
  int getScore();



  /**
   * Retrieves a list of messages providing additional information about the
   * reason for the associated state and score.
   *
   * @return  A list of messages providing additional information about the
   *          reason for the associated state and score, or an empty list if
   *          no messages are available.
   */
  List<String> getMessages();



  /**
   * Retrieves the time that this health check result was created.  The value
   * returned will be an offset in milliseconds since 12:00 a.m. on January 1,
   * 1970.
   *
   * @return  The time that this health check result was created.
   */
  long getTime();



  /**
   * Retrieves a hash code for this health check result.
   *
   * @return  A hash code for this health check result.
   */
  int hashCode();



  /**
   * Indicates whether the provided object may be considered equal to this
   * health check result.
   *
   * @param  o  The object for which to make the determination.
   *
   * @return  {@code true} if the provided object may be considered equal to
   *          this health check result, or {@code false} if not.
   */
  boolean equals(final Object o);



  /**
   * Compares the provided health check result with this health check result to
   * determine their relative order in a sorted list.  The ordering will be
   * based primarily on the health check state, and the score will be used to
   * differentiate between servers with the same state.
   *
   * @param  r  The health check result to be compared with this health check
   *            result.  It must not be {@code null}.
   *
   * @return  A negative value if this health check result should be ordered
   *          before the provided result, a positive value if this health check
   *          result should be ordered after the provided result, or zero if
   *          they are logically equivalent.
   */
  int compareTo(final HealthCheckResult r);



  /**
   * Compares the provided health check results to determine their relative
   * order in a sorted list.  The ordering will be based primarily on the health
   * check state, and the score will be used to differentiate between servers
   * with the same state.
   *
   * @param  r1  The first health check result to be compared.  It must not be
   *             {@code null}.
   * @param  r2  The second health check result to be compared.  It must not be
   *             {@code null}.
   *
   * @return  A negative value if the first health check result should be
   *          ordered before the second, a positive value if the first health
   *          check result should be ordered after the second, or zero if they
   *          are logically equivalent.
   */
  int compare(final HealthCheckResult r1, final HealthCheckResult r2);



  /**
   * Indicates whether this health check result is considered better than the
   * provided result.
   *
   * @param  r  The result for which to make the determination.  It must not be
   *            {@code null}.
   *
   * @return  {@code true} if this health check result is considered better than
   *          the provided result, or {@code false} if this health check result
   *          is worse than or equivalent to the provided result.
   */
  boolean betterThan(final HealthCheckResult r);



  /**
   * Indicates whether this health check result is considered worse than the
   * provided result.
   *
   * @param  r  The result for which to make the determination.  It must not be
   *            {@code null}.
   *
   * @return  {@code true} if this health check result is considered worse than
   *          the provided result, or {@code false} if this health check result
   *          is better than or equivalent to the provided result.
   */
  boolean worseThan(final HealthCheckResult r);



  /**
   * Retrieves a string representation of this health check result.
   *
   * @return  A string representation of this health check result.
   */
  String toString();
}
