/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2014-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.proxy.types;



import java.util.Date;

import com.unboundid.util.StaticUtils;



/**
 * This class provides a data structure for holding information about an
 * affinity to a particular backend server instance.
 */
public final class ServerAffinity
{
  // The backend server to which the affinity is established.
  private final BackendServer backendServer;

  // The time that the affinity was established.
  private final long affinityEstablishedTime;



  /**
   * Creates a server affinity for the provided backend server.
   *
   * @param  backendServer  The backend server with which the affinity is to be
   *                        associated.
   */
  public ServerAffinity(final BackendServer backendServer)
  {
    this.backendServer = backendServer;

    affinityEstablishedTime = System.currentTimeMillis();
  }



  /**
   * Retrieves the backend server to which the affinity is established.
   *
   * @return  The backend server to which the affinity is established.
   */
  public BackendServer getBackendServer()
  {
    return backendServer;
  }



  /**
   * Retrieves a timestamp indicating when the affinity was established.  The
   * timestamp value will be the number of milliseconds since the epoch, in the
   * same format returned by {@code System.currentTimeMillis} or
   * {@code Date.getTime}.
   *
   * @return  A timestamp indicating when the affinity was established.
   */
  public long getAffinityEstablishedTime()
  {
    return affinityEstablishedTime;
  }



  /**
   * Retrieves a string representation of this server affinity object.
   *
   * @return  A string representation of this server affinity object.
   */
  @Override()
  public String toString()
  {
    final StringBuilder buffer = new StringBuilder();
    toString(buffer);
    return buffer.toString();
  }



  /**
   * Appends a string representation of this server affinity object to the
   * provided buffer.
   *
   * @param  buffer  The buffer to which the information should be appended.
   */
  public void toString(final StringBuilder buffer)
  {
    buffer.append("ServerAffinity(backendServer='");
    buffer.append(backendServer.getServerAddress());
    buffer.append(':');
    buffer.append(backendServer.getServerPort());
    buffer.append("', establishedTime='");
    buffer.append(StaticUtils.encodeGeneralizedTime(
         new Date(affinityEstablishedTime)));
    buffer.append("')");
  }
}
