/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.api;

import com.unboundid.directory.sdk.common.internal.ExampleUsageProvider;
import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.common.internal.UnboundIDExtension;
import com.unboundid.directory.sdk.sync.config.KafkaSyncDestinationPluginConfig;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.directory.sdk.sync.types.KafkaMessage;
import com.unboundid.directory.sdk.sync.types.PreStepResult;
import com.unboundid.directory.sdk.sync.types.SyncOperation;
import com.unboundid.directory.sdk.sync.types.SyncServerContext;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * This class defines an API that must be implemented by extensions that
 * perform processing on synchronization operations within a Kafka Sync
 * Destination. These extensions may be used to
 * <UL>
 *   <LI>Filter out messages being published to Kafka.</LI>
 *   <LI>Modify the format or contents of messages before publishing.</LI>
 *   <LI>Modify the message key before publishing.</LI>
 * </UL>
 * <BR>
 * <H2>Configuring Kafka Sync Destination Plugins</H2>
 * In order to configure a Kafka sync destination plugin created using this API,
 * use a command like:
 * <PRE>
 *   dsconfig create-sync-destination-plugin \
 *        --plugin-name "<I>{plugin-name}</I>" \
 *        --type third-party-kafka \
 *        --set "extension-class:<I>{class-name}</I>" \
 *        --set "extension-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{plugin-name}</I>" is the name to use for the Kafka sync
 * destination plugin instance, "<I>{class-name}</I>" is the fully-
 * qualified name of the Java class that extends
 * {@code com.unboundid.directory.sdk.sync.api.KafkaSyncDestinationPlugin},
 * and "<I>{name=value}</I>" represents name-value pairs for any arguments to
 * provide to the Kafka sync destination plugin. If multiple arguments should
 * be provided to the plugin, then the "<CODE>extension-argument</CODE>"
 * option should be provided multiple times.
 */
@Extensible()
@SynchronizationServerExtension(appliesToLocalContent = false,
        appliesToSynchronizedContent = true)
@ThreadSafety(level = ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class KafkaSyncDestinationPlugin
        implements UnboundIDExtension,
        Reconfigurable<KafkaSyncDestinationPluginConfig>,
        ExampleUsageProvider
{

  /**
   * Creates a new instance of this Kafka sync destination plugin.
   * All sync destination implementations must include a default
   * constructor, but any initialization should generally be done in
   * the {@code initializeKafkaSyncDestinationPlugin} method.
   */
  public KafkaSyncDestinationPlugin()
  {
    // No default behavior.
  }



  /**
   * Retrieves a human-readable name for this extension.
   *
   * @return  A human-readable name for this extension.
   */
  @Override
  public abstract String getExtensionName();



  /**
   * Retrieves a human-readable description for this extension.  Each element
   * of the array that is returned will be considered a separate paragraph in
   * generated documentation.
   *
   * @return  A human-readable description for this extension, or {@code null}
   *          or an empty array if no description should be available.
   */
  @Override
  public abstract String[] getExtensionDescription();



  /**
   * Updates the provided argument parser to define any configuration arguments
   * which may be used by this extension.  The argument parser may also be
   * updated to define relationships between arguments (e.g., to specify
   * required, exclusive, or dependent argument sets).
   *
   * @param  parser  The argument parser to be updated with the configuration
   *                 arguments which may be used by this extension.
   *
   * @throws  ArgumentException  If a problem is encountered while updating the
   *                             provided argument parser.
   */
  @Override
  public void defineConfigArguments(final ArgumentParser parser)
          throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initialized this Kafka sync destination plugin. This method will be
   * called before any other methods in the class.
   *
   * @param  serverContext  A handle to the server context for the
   *                        ${SYNC_SERVER_BASE_NAME} in which this extension is
   *                        running. Extensions should typically store this
   *                        in a class member.
   * @param  config         The general configuration for this plugin.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this Kafka sync destination
   *                        plugin.
   */
  public void initializeKafkaSyncDestinationPlugin(
          final SyncServerContext serverContext,
          final KafkaSyncDestinationPluginConfig config,
          final ArgumentParser parser)
  {
    // No Initialization will be performed by default.
  }



  /**
   * Indicates whether the configuration represented by the provided argument
   * parser is acceptable for use by this extension.  The parser will have been
   * used to parse any configuration available for this extension, and any
   * automatic validation will have been performed.  This method may be used to
   * perform any more complex validation which cannot be performed automatically
   * by the argument parser.
   *
   * @param  config               The general configuration for this extension.
   * @param  parser               The argument parser that has been used to
   *                              parse the proposed configuration for this
   *                              extension.
   * @param  unacceptableReasons  A list to which messages may be added to
   *                              provide additional information about why the
   *                              provided configuration is not acceptable.
   *
   * @return  {@code true} if the configuration in the provided argument parser
   *          appears to be acceptable, or {@code false} if not.
   */
  @Override
  public boolean isConfigurationAcceptable(
          final KafkaSyncDestinationPluginConfig config,
          final ArgumentParser parser,
          final List<String> unacceptableReasons)
  {
    // No extended validation will be performed by default.
    return true;
  }



  /**
   * Attempts to apply the configuration from the provided argument parser to
   * this extension.
   *
   * @param  config                The general configuration for this extension.
   * @param  parser                The argument parser that has been used to
   *                               parse the new configuration for this
   *                               extension.
   * @param  adminActionsRequired  A list to which messages may be added to
   *                               provide additional information about any
   *                               additional administrative actions that may
   *                               be required to apply some of the
   *                               configuration changes.
   * @param  messages              A list to which messages may be added to
   *                               provide additional information about the
   *                               processing performed by this method.
   *
   * @return  A result code providing information about the result of applying
   *          the configuration change.  A result of {@code SUCCESS} should be
   *          used to indicate that all processing completed successfully.  Any
   *          other result will indicate that a problem occurred during
   *          processing.
   */
  @Override
  public ResultCode applyConfiguration(
          final KafkaSyncDestinationPluginConfig config,
          final ArgumentParser parser,
          final List<String> adminActionsRequired,
          final List<String> messages)
  {
    // By default, no configuration changes will be applied.
    return ResultCode.SUCCESS;
  }



  /**
   * Performs any cleanup that may be necessary when this Kafka sync destination
   * plugin is taken out of service.  This can happen when it is deleted from
   * the configuration and at server shutdown.
   */
  public void finalizeKafkaSyncDestinationPlugin()
  {
    // No implementation is required.
  }

  /**
   * Provides the ability to alter or suppress Kafka messages.
   * This method is called before a message is published to the
   * configured Kafka Topic.
   *
   * @param message     The message to be published to the
   *                    Kafka Topic.
   * @param operation   The synchronization operation for this
   *                    change.
   *
   * @return  The result of the plugin processing.
   */
  public PreStepResult prePublish(final KafkaMessage message,
                                  final SyncOperation operation)
  {
    return PreStepResult.CONTINUE;
  }

  /**
   * Retrieves a string representation of this Kafka sync destination plugin.
   *
   * @return  A string representation of this Kafka sync destination plugin.
   */
  @Override()
  public final String toString()
  {
    final StringBuilder buffer = new StringBuilder();
    toString(buffer);
    return buffer.toString();
  }

  /**
   * Appends a string representation of this Kafka sync destination plugin to
   * the provided buffer.
   *
   * @param  buffer  The buffer to which the string representation should be
   *                 appended.
   */
  public abstract void toString(final StringBuilder buffer);

  /**
   * Retrieves a map containing examples of configurations that may be used for
   * this extension.  The map key should be a list of sample arguments, and the
   * corresponding value should be a description of the behavior that will be
   * exhibited by the extension when used with that configuration.
   *
   * @return  A map containing examples of configurations that may be used for
   *          this extension.  It may be {@code null} or empty if there should
   *          not be any example argument sets.
   */
  @Override
  public Map<List<String>,String> getExamplesArgumentSets()
  {
    return Collections.emptyMap();
  }
}
