/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.scripting;



import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import com.unboundid.directory.sdk.common.internal.Reconfigurable;
import com.unboundid.directory.sdk.sync.config.LDAPSyncSourcePluginConfig;
import com.unboundid.directory.sdk.sync.internal.SynchronizationServerExtension;
import com.unboundid.directory.sdk.sync.types.PostStepResult;
import com.unboundid.directory.sdk.sync.types.SyncOperation;
import com.unboundid.directory.sdk.sync.types.SyncServerContext;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPInterface;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.util.Extensible;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.args.ArgumentException;
import com.unboundid.util.args.ArgumentParser;



/**
 * This class defines an API that must be implemented by scripted extensions
 * that perform processing on synchronization operations within an LDAP Sync
 * Source.  These extensions may be used to
 * <ul>
 *   <li>Filter out certain changes from being synchronized.</li>
 *   <li>Change how an entry is fetched.</li>
 * </ul>
 * <BR>
 * A note on exception handling: in general subclasses should not
 * catch LDAPExceptions that are thrown when using the provided
 * LDAPInterface unless there are specific exceptions that are
 * expected.  The ${SYNC_SERVER_BASE_NAME} will handle
 * LDAPExceptions in an appropriate way based on the specific
 * cause of the exception.  For example, some errors will result
 * in the SyncOperation being retried, and others will trigger
 * fail over to a different server.
 * <BR>
 * <H2>Configuring Groovy-Scripted LDAP Sync Source Plugins</H2>
 * In order to configure a scripted LDAP sync source plugin based on this API
 * and written in the Groovy scripting language, use a command like:
 * <PRE>
 *      dsconfig create-sync-source-plugin \
 *           --plugin-name "<I>{plugin-name}</I>" \
 *           --type groovy-scripted-ldap \
 *           --set "script-class:<I>{class-name}</I>" \
 *           --set "script-argument:<I>{name=value}</I>"
 * </PRE>
 * where "<I>{plugin-name}</I>" is the name to use for the LDAP sync source
 * plugin instance, "<I>{class-name}</I>" is the fully-qualified name of the
 * Groovy class written using this API, and "<I>{name=value}</I>" represents
 * name-value pairs for any arguments to provide to the LDAP sync source plugin.
 * If multiple arguments should be provided to the LDAP sync source plugin, then
 * the "<CODE>--set script-argument:<I>{name=value}</I></CODE>" option should be
 * provided multiple times.
 *
 * @see  com.unboundid.directory.sdk.sync.api.LDAPSyncSourcePlugin
 */
@Extensible()
@SynchronizationServerExtension(appliesToLocalContent=false,
     appliesToSynchronizedContent=true)
@ThreadSafety(level= ThreadSafetyLevel.INTERFACE_THREADSAFE)
public abstract class ScriptedLDAPSyncSourcePlugin
       implements Reconfigurable<LDAPSyncSourcePluginConfig>
{
  /**
   * Creates a new instance of this LDAP sync source plugin.  All sync
   * source implementations must include a default constructor, but any
   * initialization should generally be done in the
   * {@code initializeLDAPSyncSourcePlugin} method.
   */
  public ScriptedLDAPSyncSourcePlugin()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public void defineConfigArguments(final ArgumentParser parser)
         throws ArgumentException
  {
    // No arguments will be allowed by default.
  }



  /**
   * Initializes this LDAP sync source plugin.
   *
   * @param  serverContext  A handle to the server context for the server in
   *                        which this extension is running.
   * @param  config         The general configuration for this LDAP sync
   *                        source plugin transformation.
   * @param  parser         The argument parser which has been initialized from
   *                        the configuration for this LDAP sync source plugin.
   *
   * @throws  LDAPException  If a problem occurs while initializing this LDAP
   *                         sync source plugin.
   */
  public void initializeLDAPSyncSourcePlugin(
                   final SyncServerContext serverContext,
                   final LDAPSyncSourcePluginConfig config,
                   final ArgumentParser parser)
         throws LDAPException
  {
    // No initialization will be performed by default.
  }



  /**
   * Performs any cleanup which may be necessary when this LDAP sync source
   * plugin is to be taken out of service.
   */
  public void finalizeLDAPSyncSourcePlugin()
  {
    // No implementation is required.
  }



  /**
   * {@inheritDoc}
   */
  public boolean isConfigurationAcceptable(
                      final LDAPSyncSourcePluginConfig config,
                      final ArgumentParser parser,
                      final List<String> unacceptableReasons)
  {
    // No extended validation will be performed.
    return true;
  }



  /**
   * {@inheritDoc}
   */
  public ResultCode applyConfiguration(final LDAPSyncSourcePluginConfig config,
                         final ArgumentParser parser,
                         final List<String> adminActionsRequired,
                         final List<String> messages)
  {
    // By default, no configuration changes will be applied.
    return ResultCode.SUCCESS;
  }



  /**
   * This method is called after fetching a source entry.  A
   * connection to the source server is provided.  This method is
   * overridden by plugins that need to manipulate the search results that
   * are returned to the Sync Pipe.  This can include filtering out certain
   * entries, remove information from the entries, or adding additional
   * information, possibly by doing a followup LDAP search.
   *
   * @param  sourceConnection       A connection to the source server.
   * @param  fetchedEntryRef        A reference to the entry that was fetched.
   *                                This entry can be edited in place, or the
   *                                reference can be changed to point to a
   *                                different entry that the plugin constructs.
   * @param  operation              The synchronization operation for this
   *                                change.
   *
   * @return  The result of the plugin processing.
   *
   * @throws  LDAPException  In general subclasses should not catch
   *                         LDAPExceptions that are thrown when
   *                         using the LDAPInterface unless there
   *                         are specific exceptions that are
   *                         expected.  The ${SYNC_SERVER_BASE_NAME}
   *                         will handle LDAPExceptions in an
   *                         appropriate way based on the specific
   *                         cause of the exception.  For example,
   *                         some errors will result in the
   *                         SyncOperation being retried, and others
   *                         will trigger fail over to a different
   *                         server.  Plugins should only throw
   *                         LDAPException for errors related to
   *                         communication with the LDAP server.
   *                         Use the return code to indicate other
   *                         types of errors, which might require
   *                         retry.
   */
  public PostStepResult postFetch(final LDAPInterface sourceConnection,
                                  final AtomicReference<Entry> fetchedEntryRef,
                                  final SyncOperation operation)
       throws LDAPException
  {
    return PostStepResult.CONTINUE;
  }
}
