/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.types;

import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.ldap.sdk.ChangeType;
import com.unboundid.util.InternalUseOnly;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import com.unboundid.directory.sdk.sync.util.ScriptUtils;

/**
 * This class represents the basis for a single change record. This is
 * effectively a hint that a change happened, and some metadata about the
 * change. A SyncSource implementation should create instances of this class
 * based on changes detected in the source endpoint (either from a changelog
 * table or some other change tracking mechanism). The resync process will also
 * use instances of this class to identify source entries, which can then be
 * fetched using the <code>fetchEntry()</code> method on the SyncSource
 * extension.
 */
@ThreadSafety(level = ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class ChangeRecord
{

  // The basic, common attributes of a generic ChangeRecord.
  private final long changeNumber;
  private final ChangeType changeType;
  private final DN identifiableInfo;
  private final DN identifiableInfoAfterChange;
  private final String[] changedAttributes;
  private final String modifier;
  private final long changeTime;
  private final Entry fullEntry;
  private final Map<Object, Object> properties;
  private CompletionStatus completionStatus;

  private static final AtomicLong uniqueIdSequence = new AtomicLong(0);

  /**
   * Private constructor, uses Builder to construct immutable instance.
   * @param bldr the Builder from which to create this change record.
   */
  private ChangeRecord(final Builder bldr)
  {
    // if changeNumber is not set, assign it an auto-incrementing value. we need
    // this to have a way to identify a specific change and the order in which
    // changes are processed
    changeNumber = (bldr.changeNumber == -1) ?
            uniqueIdSequence.getAndIncrement() : bldr.changeNumber;
    changeType = bldr.changeType;
    identifiableInfo = bldr.identifiableInfo;
    identifiableInfoAfterChange = bldr.changeType == ChangeType.MODIFY_DN
            ? bldr.identifiableInfoAfterChange : identifiableInfo;
    changedAttributes = bldr.changedAttributes;
    modifier = bldr.modifier;
    changeTime = bldr.changeTime;
    fullEntry = bldr.fullEntry;
    properties = bldr.properties;
  }

  /**
   * Get the change number that identifies this particular change. If a change
   * number is not used by the source for change detection, this method will
   * return a monotonically increasing sequence number for this change record,
   * so that you can still identify the order in which changes were detected.
   * @return the changeNumber
   */
  public long getChangeNumber()
  {
    return changeNumber;
  }

  /**
   * Get the change type (ADD/MODIFY/MOD-DN/DELETE). For <i>resync</i>
   * operations, this will be <code>null</code>.
   * @return the changeType
   */
  public ChangeType getChangeType()
  {
    return changeType;
  }

  /**
   * Get a DN that identifies the entry or record that changed (for example
   * "accountID=123"). If multiple attributes are part of the identifier,
   * they will be represented as different
   * RDN components of the DN (for example "accountID=123,groupID=5").
   * @return an identifier DN for the entry that changed
   */
  public DN getIdentifiableInfo()
  {
    return identifiableInfo;
  }

  /**
   * Get a DN that identifies the entry or record after the change is
   * complete. This DN will only differ from the result of
   * {@code getIdentifiableInfo()} if the change type was a Modify DN.
   * @return the DN for the entry after the change.
   */
  public DN getIdentifiableInfoAfterChange() {
    return identifiableInfoAfterChange;
  }

  /**
   * Get the set of changed attributes for this change.
   * @return an array of attribute names that were modified as part of the
   *         change
   */
  public String[] getChangedAttributes()
  {
    return changedAttributes;
  }

  /**
   * Get the user account name that made the change.
   * @return the source user account
   */
  public String getModifier()
  {
    return modifier;
  }

  /**
   * Get the time at which the change occurred.
   * @return the change time (in milliseconds since
   *         January 1, 1970 00:00:00.000 GMT)
   */
  public long getChangeTime()
  {
    return changeTime;
  }

  /**
   * Get the full source entry (if it was set on this ChangeRecord when it was
   * created). Typically this will be <code>null</code>, but some extensions
   * may opt to set the entry when the ChangeRecord is constructed in order to
   * skip the <code>fetchEntry()</code> phase of processing.
   *
   * @return the full source entry, or <code>null</code> if it has not been set
   */
  public Entry getFullEntry()
  {
    return fullEntry;
  }

  /**
   * Get the property value (if one exists) for the given key.
   * @param key the key for a given property to return
   * @return the property value, or <code>null</code> if the key is null
   */
  public Object getProperty(final Object key)
  {
    if(key == null)
    {
      return null;
    }
    return properties.get(key);
  }

  /**
   * This method is used by the Sync Pipe to indicate if the completion status
   * of a synchronization operation. This is for internal use only
   * and should not be called by extension code.
   * @param status the completion status for this ChangeRecord
   */
  @InternalUseOnly
  public void setCompletionStatus(final CompletionStatus status)
  {
    this.completionStatus = status;
  }

  /**
   * Gets the completion status for this change. This will be null if the change
   * has not finished processing yet.
   * @return the CompletionStatus indicating whether this change completed
   *          successfully or else a reason why it failed
   */
  public CompletionStatus getCompletionStatus()
  {
    return completionStatus;
  }

  /**
   * This class is used to construct ChangeRecord instances. At least a
   * {@link ChangeType} and an identifiableInfo {@link DN} are required; the
   * rest of the parameters are optional.
   * Arbitrary properties can also be added to the object by calling
   * {@link #addProperty(Object, Object)}. The setter methods return the Builder
   * instance itself, so that these calls can be chained. When finished setting
   * up parameters, call the {@link #build()} method to create a new
   * {@link ChangeRecord}.
   */
  public static class Builder
  {
    // required parameters
    private final ChangeType changeType;
    private final DN identifiableInfo;

    // optional parameters
    private long changeNumber;
    private String[] changedAttributes;
    private String modifier;
    private long changeTime;
    private Entry fullEntry;
    private DN identifiableInfoAfterChange;

    // various other values that are attached to the change record
    private final Map<Object, Object> properties =
            new ConcurrentHashMap<Object, Object>();

    // flag to indicate whether this builder has been built
    private volatile boolean built;

    /**
     * Creates a Builder which can be used to construct a ChangeRecord.
     * @param type
     *          the ChangeType (ADD/MODIFY/MOD-DN/DELETE). This can be
     *          <code>null</code> to indicate a resync operation.
     * @param identifiableInfo
     *          a unique identifier for the entry that changed
     *          (i.e. "accountID=123"). If multiple attributes are part of
     *          the identifier, they should be separate RDN components of the DN
     *          (i.e. "accountID=123,groupID=5").
     */
    public Builder(final ChangeType type, final DN identifiableInfo)
    {
      if(identifiableInfo == null)
      {
        throw new IllegalArgumentException(
              "The 'identifiableInfo' parameter cannot be null.");
      }
      this.changeType = type;
      this.identifiableInfo = identifiableInfo;
      this.changeNumber = -1;
      this.built = false;
    }

    /**
     * Creates a Builder which can be used to construct a ChangeRecord.
     * @param type
     *          the ChangeType (ADD/MODIFY/MOD-DN/DELETE). This can be
     *          <code>null</code> to indicate a resync operation.
     * @param identifiableInfo
     *          a unique identifier for the entry that changed
     *          (i.e. "accountID=123"). If multiple attributes are part of
     *          the identifier, they should be delimited with the default
     *          delimiter of "%%" (i.e. "accountID=123%%groupID=5").
     */
    public Builder(final ChangeType type, final String identifiableInfo)
    {
      this(type, ScriptUtils.idStringToDN(identifiableInfo, null));
    }

    /**
     * Creates a Builder which can be used to construct a ChangeRecord.
     * @param type
     *          the ChangeType (ADD/MODIFY/MOD-DN/DELETE). This can be
     *          <code>null</code> to indicate a resync operation.
     * @param identifiableInfo
     *          a unique identifier for the row that changed
     *          (i.e. "accountID=123"). If multiple attributes are part of
     *          the identifier, they should be delimited with a unique string
     *          (i.e. "accountID=123%%groupID=5") which is specified by the
     *          <i>delimiter</i> parameter.
     * @param delimiter
     *          The delimiter used to split separate components of the
     *          identifiable info. If this is null, the default of "%%" will be
     *          used.
     */
    public Builder(final ChangeType type, final String identifiableInfo,
                    final String delimiter)
    {
      this(type, ScriptUtils.idStringToDN(identifiableInfo, delimiter));
    }

    /**
     * Set the change number that identifies this particular change (if
     * applicable). If this is not used, a change number will be automatically
     * generated for the ChangeRecord.
     * @param changeNumber
     *          the change number
     * @return the Builder instance
     */
    public Builder changeNumber(final long changeNumber)
    {
      this.changeNumber = changeNumber;
      return this;
    }

    /**
     * Set the DN of the entry after the change. This should only be used if
     * the change type was a Modify DN.
     * @param   identifiableInfoAfterChange The final DN of the entry after
     *                                      the change.
     * @return                              the Builder instance.
     * @throws  IllegalArgumentException if the change type is not Modify DN.
     */
    public Builder identifiableInfoAfterChange(
            final DN identifiableInfoAfterChange)
    {
      if (changeType != ChangeType.MODIFY_DN) {
        throw new IllegalArgumentException("identifiableInfoAfterChange can " +
                "only be set for Modify DN changes.");
      }
      this.identifiableInfoAfterChange = identifiableInfoAfterChange;
      return this;
    }

    /**
     * Set the set of changed attributes for this change entry.
     * @param changedAttributes
     *          an array of attribute names that were modified as part of the
     *          change
     * @return the Builder instance
     */
    public Builder changedAttributes(final String[] changedAttributes)
    {
      this.changedAttributes = changedAttributes;
      return this;
    }

    /**
     * Set the user account name that made the change.
     * @param modifier
     *          the account name or user name of the entity that made the change
     * @return the Builder instance
     */
    public Builder modifier(final String modifier)
    {
      this.modifier = modifier;
      return this;
    }

    /**
     * Set the time at which the change occurred. This should be based on the
     * clock at the source endpoint if possible.
     * @param changeTime the time of the change (in milliseconds since
                         January 1, 1970 00:00:00.000 GMT)
     * @return the Builder instance
     */
    public Builder changeTime(final long changeTime)
    {
      this.changeTime = changeTime;
      return this;
    }

    /**
     * Set the full source entry on this ChangeRecord. This may be desirable if
     * the source does not provide logical separation between the
     * "change record" and the entry itself. If this is set on the ChangeRecord,
     * the ${SYNC_SERVER_BASE_NAME} will skip the call to
     * <code>fetchEntry()</code> in your extension and instead use this
     * {@link Entry}.
     * <p>
     * When using this mechanism, make sure to set it to a non-null Entry even
     * on a DELETE, because this will be used to correlate to the destination
     * entry to delete.
     * @param entry the full source entry that was changed
     * @return the Builder instance
     */
    public Builder fullEntry(final Entry entry)
    {
      this.fullEntry = entry;
      return this;
    }

    /**
     * Add an arbitrary attachment or property to the ChangeRecord being
     * built. Nor the key or the value are allowed to be <code>null</code>.
     * @param key
     *          the key for the property
     * @param value
     *          the value of the property
     * @return the Builder instance
     */
    public Builder addProperty(final Object key, final Object value)
    {
      this.properties.put(key, value);
      return this;
    }

    /**
     * Construct the ChangeRecord. This method may only be called once.
     * @return a ChangeRecord instance
     */
    public synchronized ChangeRecord build()
    {
      if(built)
      {
        throw new IllegalStateException("This Builder has already been built.");
      }
      built = true;
      return new ChangeRecord(this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString()
  {
    String changeTypeStr = changeType != null ? changeType.toString() :"resync";
    String fullEntryStr = fullEntry != null ?
              ", fullEntry=" + fullEntry.toLDIFString() : "";
    return "ChangeRecord [changeNumber=" + changeNumber +
            ", changeType=" + changeTypeStr +
            ", identifiableInfo=" + identifiableInfo +
            ", identifiableInfoAfterChange=" + identifiableInfoAfterChange +
            ", changedAttributes=" + Arrays.toString(changedAttributes) +
            ", modifier=" + modifier + ", changeTime=" +
            (new Date(changeTime)).toString() +
            ", properties=" + properties + fullEntryStr + "]";
  }
}
