/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.types;

import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import com.unboundid.ldap.sdk.DN;
import com.unboundid.util.InternalUseOnly;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import com.unboundid.directory.sdk.sync.util.ScriptUtils;

/**
 * This class represents the basis for a single database change record. A
 * JDBCSyncSource implementation should create instances of this
 * class based on changes detected in the database (either from a changelog
 * table or some other change tracking mechanism). The resync process will also
 * use instances of this class to identify database entries, which can then be
 * fetched using the <code>fetchEntry()</code> method on the JDBCSyncSource
 * extension.
 */
@ThreadSafety(level = ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class DatabaseChangeRecord
{

  /** Potential types of database changes. */
  public enum ChangeType
  {
    /**
     * Represents a database insert.
     */
    insert,
    /**
     * Represents a database delete.
     */
    delete,
    /**
     * Represents a database update.
     */
    update,
    /**
     * Represents a resync operation. This is a special type used by the resync
     * command-line tool. See
     * <code>JDBCSyncSource#listAllEntries()</code> for more
     * information.
     */
    resync
  }

  // the basic, common attributes of a DatabaseChangeRecord.
  private final long changeNumber;
  private final ChangeType changeType;
  private final String tableName;
  private final DN identifiableInfo;
  private final String entryType;
  private final String[] changedColumns;
  private final String modifier;
  private final long changeTime;
  private final Map<Object, Object> properties;
  private CompletionStatus completionStatus;

  private static final AtomicLong uniqueIdSequence = new AtomicLong(0);

  /**
   * Private constructor, uses Builder to construct immutable instance.
   * @param bldr the Builder from which to create this change record.
   */
  private DatabaseChangeRecord(final Builder bldr)
  {
    // if changeNumber is not set, assign it an auto-incrementing value. we need
    // this to have a way to identify a specific change and the order in which
    // changes are processed
    changeNumber = (bldr.changeNumber == -1) ?
            uniqueIdSequence.getAndIncrement() : bldr.changeNumber;
    changeType = bldr.changeType;
    identifiableInfo = bldr.identifiableInfo;
    tableName = bldr.tableName;
    entryType = bldr.entryType;
    changedColumns = bldr.changedColumns;
    modifier = bldr.modifier;
    changeTime = bldr.changeTime;
    properties = bldr.properties;
  }

  /**
   * Get the change number that identifies this particular change. If a change
   * number is not used by the database for change detection, this method will
   * return a monotonically increasing sequence number for this change record,
   * so that you can still identify the order in which changes were detected.
   * @return the changeNumber
   */
  public long getChangeNumber()
  {
    return changeNumber;
  }

  /**
   * Get the change type (insert/update/delete/resync).
   * @return the changeType
   */
  public ChangeType getChangeType()
  {
    return changeType;
  }

  /**
   * Get the database table on which the change occurred.
   * @return the table name
   */
  public String getTableName()
  {
    return tableName;
  }

  /**
   * Get the DN that identifies the row that changed (for example
   * "accountID=123"). If multiple attributes are part of the identifier,
   * they will be represented as different
   * RDN components of the DN in the order they were originally specified
   * (for example "accountID=123,groupID=5").
   * @return an identifier string
   */
  public DN getIdentifiableInfo()
  {
    return identifiableInfo;
  }

  /**
   * Get the database entry type that this change corresponds to (for example
   * "account" or "subscriber").
   * @return the type of database entry
   */
  public String getEntryType()
  {
    return entryType;
  }

  /**
   * Get the set of changed columns for this change entry.
   * @return an array of column names that were modified as part of the change
   */
  public String[] getChangedColumns()
  {
    return changedColumns;
  }

  /**
   * Get the database user that made the change.
   * @return the database user account
   */
  public String getModifier()
  {
    return modifier;
  }

  /**
   * Get the time at which the change occurred.
   * @return the change time (in milliseconds since
   *                          January 1, 1970 00:00:00.000 GMT)
   */
  public long getChangeTime()
  {
    return changeTime;
  }

  /**
   * Get the property value (if one exists) for the given key.
   * @param key the property key whose value to return
   * @return the property value, or <code>null</code> if the key is null
   */
  public Object getProperty(final Object key)
  {
    if(key == null)
    {
      return null;
    }
    return properties.get(key);
  }

  /**
   * This method is used by the Sync Pipe to indicate if the completion status
   * of a synchronization operation. This is for internal use only
   * and should not be called by clients.
   * @param status the completion status for this DatabaseChangeRecord
   */
  @InternalUseOnly
  public void setCompletionStatus(final CompletionStatus status)
  {
    this.completionStatus = status;
  }

  /**
   * Gets the completion status for this change. This will be null if the change
   * has not finished processing yet.
   * @return the CompletionStatus indicating whether this change completed
   *          successfully or else a reason why it failed
   */
  public CompletionStatus getCompletionStatus()
  {
    return completionStatus;
  }

  /**
   * This class is used to construct DatabaseChangeRecord instances. At least a
   * {@link ChangeType} and an identifiableInfo {@link DN} are required; the
   * rest of the parameters are optional.
   * Arbitrary properties can also be added to the object by calling
   * {@link #addProperty(Object, Object)}. The setter methods return the Builder
   * instance itself, so that these calls can be chained. When finished setting
   * up parameters, call the {@link #build()} method to create a new
   * {@link DatabaseChangeRecord}.
   */
  public static class Builder
  {
    // required parameters
    private final ChangeType changeType;
    private final DN identifiableInfo;

    // optional parameters
    private long changeNumber;
    private String tableName;
    private String entryType;
    private String[] changedColumns;
    private String modifier;
    private long changeTime;

    // various other values that are attached to the change record
    private final Map<Object, Object> properties =
            new ConcurrentHashMap<Object, Object>();

    // flag to indicate whether this builder has been built
    private volatile boolean built;

    /**
     * Creates a Builder which can be used to construct a DatabaseChangeRecord.
     * @param type
     *          the ChangeType (insert/update/delete/resync)
     * @param identifiableInfo
     *          a unique identifier for the row that changed
     *          (i.e. "accountID=123"). If multiple attributes are part of
     *          the identifier, they should be separate RDN components of the DN
     *          (i.e. "accountID=123,groupID=5").
     */
    public Builder(final ChangeType type, final DN identifiableInfo)
    {
      if(type == null || identifiableInfo == null)
      {
        throw new IllegalArgumentException(
              "The 'type' and 'identifiableInfo' parameters can not be null.");
      }
      this.changeType = type;
      this.identifiableInfo = identifiableInfo;
      this.changeNumber = -1;
      this.built = false;
    }

    /**
     * Creates a Builder which can be used to construct a DatabaseChangeRecord.
     * @param type
     *          the ChangeType (insert/update/delete/resync)
     * @param identifiableInfo
     *          a unique identifier for the row that changed
     *          (i.e. "accountID=123"). If multiple attributes are part of
     *          the identifier, they should be delimited with the default
     *          delimiter of "%%" (i.e. "accountID=123%%groupID=5").
     */
    public Builder(final ChangeType type, final String identifiableInfo)
    {
      this(type, ScriptUtils.idStringToDN(identifiableInfo, null));
    }

    /**
     * Creates a Builder which can be used to construct a DatabaseChangeRecord.
     * @param type
     *          the ChangeType (insert/update/delete/resync)
     * @param identifiableInfo
     *          a unique identifier for the row that changed
     *          (i.e. "accountID=123"). If multiple attributes are part of
     *          the identifier, they should be delimited with a unique string
     *          (i.e. "accountID=123%%groupID=5") which is specified by the
     *          <i>delimiter</i> parameter.
     * @param delimiter
     *          The delimiter used to split separate components of the
     *          identifiable info. If this is null, the default of "%%" will be
     *          used.
     */
    public Builder(final ChangeType type, final String identifiableInfo,
                    final String delimiter)
    {
      this(type, ScriptUtils.idStringToDN(identifiableInfo, delimiter));
    }

    /**
     * Set the change number that identifies this particular change (if
     * applicable).
     * @param changeNumber the change number
     * @return the Builder instance
     */
    public Builder changeNumber(final long changeNumber)
    {
      this.changeNumber = changeNumber;
      return this;
    }

    /**
     * Set the database table on which the change occurred.
     * @param tableName the table name
     * @return the Builder instance
     */
    public Builder tableName(final String tableName)
    {
      this.tableName = tableName;
      return this;
    }

    /**
     * Set the database entry type that this change corresponds to (for example
     * "account"
     * or "subscriber").
     * @param entryType the type of database entry
     * @return the Builder instance
     */
    public Builder entryType(final String entryType)
    {
      this.entryType = entryType;
      return this;
    }

    /**
     * Set the set of changed columns for this change entry.
     * @param changedColumns
     *          an array of column names that were modified as part of the
     *          change
     * @return the Builder instance
     */
    public Builder changedColumns(final String[] changedColumns)
    {
      this.changedColumns = changedColumns;
      return this;
    }

    /**
     * Set the database user that made the change.
     * @param modifier the database account name
     * @return the Builder instance
     */
    public Builder modifier(final String modifier)
    {
      this.modifier = modifier;
      return this;
    }

    /**
     * Set the time at which the change occurred. This should be based on the
     * clock on the database server.
     * @param changeTime the time of the change (in milliseconds since
     *                   January 1, 1970 00:00:00.000 GMT)
     * @return the Builder instance
     */
    public Builder changeTime(final long changeTime)
    {
      this.changeTime = changeTime;
      return this;
    }

    /**
     * Add an arbitrary attachment or property to the DatabaseChangeRecord being
     * built. Nor the key or the value are allowed to be <code>null</code>.
     * @param key
     *          the key for the property
     * @param value
     *          the value of the property
     * @return the Builder instance
     */
    public Builder addProperty(final Object key, final Object value)
    {
      this.properties.put(key, value);
      return this;
    }

    /**
     * Construct the DatabaseChangeRecord. This method may only be called once.
     * @return a DatabaseChangeRecord instance
     */
    public synchronized DatabaseChangeRecord build()
    {
      if(built)
      {
        throw new IllegalStateException("This Builder has already been built.");
      }
      built = true;
      return new DatabaseChangeRecord(this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString()
  {
    return "DatabaseChangeRecord [changeNumber=" + changeNumber +
            ", changeType=" +
            changeType + ", tableName=" + tableName + ", identifiableInfo=" +
            identifiableInfo +
            ", entryType=" + entryType +
            ", changedColumns=" + Arrays.toString(changedColumns) +
            ", modifier=" + modifier + ", changeTime=" +
            (new Date(changeTime)).toString() +
            ", properties=" + properties + "]";
  }
}
