/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.types;

import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.util.Extensible;


/**
 * This class provides a type of {@link Exception} which is used to indicate
 * an error when interfacing with an external resource. There is an included
 * {@link PostStepResult} which can be used to communicate back to the
 * ${SYNC_SERVER_BASE_NAME} what it should do after a failure.
 */
@Extensible()
public class EndpointException extends Exception
{
  //Generated servial version UID
  private static final long serialVersionUID = -5331807629501618293L;

  //Retry policy associated with this exception
  private final PostStepResult retryPolicy;


  /**
   * Creates a new EndpointException with the provided information.
   *
   * @param  result   The result object which indicates how the
   *                  ${SYNC_SERVER_BASE_NAME} should proceed with the sync
   *                  operation. The value of {@link PostStepResult#CONTINUE}
   *                  is undefined in this context and cannot be used.
   */
  public EndpointException(final PostStepResult result)
  {
    this(result, null, null);
  }



  /**
   * Creates a new EndpointException with the provided information.
   *
   * @param  result   The result object which indicates how the
   *                  ${SYNC_SERVER_BASE_NAME} should proceed with the sync
   *                  operation. The value of {@link PostStepResult#CONTINUE}
   *                  is undefined in this context and cannot be used.
   * @param  message  The message that explains the problem that occurred.
   */
  public EndpointException(final PostStepResult result, final String message)
  {
    this(result, message, null);
  }



  /**
   * Creates a new EndpointException with the provided information.
   *
   * @param  result   The result object which indicates how the
   *                  ${SYNC_SERVER_BASE_NAME} should proceed with the sync
   *                  operation. The value of {@link PostStepResult#CONTINUE}
   *                  is undefined in this context and cannot be used.
   * @param  cause    The underlying cause that triggered this exception.
   */
  public EndpointException(final PostStepResult result, final Throwable cause)
  {
    this(result, null, cause);
  }



  /**
   * Creates a new EndpointException with the provided information.
   *
   * @param  result   The result object which indicates how the
   *                  ${SYNC_SERVER_BASE_NAME} should proceed with the sync
   *                  operation. The value of {@link PostStepResult#CONTINUE}
   *                  is undefined in this context and cannot be used.
   * @param  message  The message that explains the problem that occurred.
   * @param  cause    The underlying cause that triggered this exception.
   */
  public EndpointException(final PostStepResult result,
                           final String message,
                           final Throwable cause)
  {
    super(message, cause);
    this.retryPolicy = result;
    if(result == null)
    {
      throw new NullPointerException("The PostStepResult cannot be null.");
    }
    else if(PostStepResult.CONTINUE.equals(result))
    {
      throw new IllegalArgumentException("The PostStepResult parameter cannot" +
                                          "be CONTINUE.");
    }
  }


  /**
   * Creates a new EndpointException with the provided information. No
   * PostStepResult is required here because the ${SYNC_SERVER_BASE_NAME} has
   * built-in logic to interpret an {@link LDAPException}.
   *
   * @param cause The underlying cause that triggered this exception.
   */
  public EndpointException(final LDAPException cause)
  {
    super(cause);
    this.retryPolicy = null;
  }


  /**
   * Gets the result object which indicates how the ${SYNC_SERVER_BASE_NAME}
   * should proceed with the sync operation. The value will never be
   * {@link PostStepResult#CONTINUE}, because this is undefined in this context.
   * If the object was constructed with
   * {@link #EndpointException(LDAPException)} then this method will return
   * null.
   *
   * @return a {@link PostStepResult} indicating the retry policy that should
   *         be taken
   */
  public final PostStepResult getPostStepResult() {
    return retryPolicy;
  }
}
