/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.types;

import java.io.Serializable;
import java.util.Date;

/**
 * Used to identify how to set the starting point for synchronization (beginning
 * of changelog, end of changelog, based on a change sequence number, etc).
 * Different startpoint types require different supporting values; for example
 * if the type is RESUME_AT_CHANGE_NUMBER, the value must be a <code>long</code>
 * that represents a change number to start at. Some of the startpoint types
 * require a server name to be specified as well.
 * <p>
 * These objects will be constructed by the <code>realtime-sync</code> CLI and
 * passed down into the SetStartpointTask, which will then delegate to the
 * specific Sync Pipe and finally the Sync Source implementation for that pipe.
 * Not all Sync Source implementations will support all the startpoint types. If
 * an unsupported type is specified, the underlying implementation is free to
 * throw a {@link RuntimeException} stating that the startpoint type is
 * unsupported.
 */
public final class SetStartpointOptions
{

  /**
   * The possible startpoint types.
   */
  public enum StartpointType
  {
    /**
     * Set the startpoint to a specific change sequence number in the change
     * log.
     * Note: This type is specific to directory servers.
     */
    RESUME_AT_CSN,
    /**
     * Set the startpoint to a specific change number in the change log.
     */
    RESUME_AT_CHANGE_NUMBER,
    /**
     * Set the startpoint to a specific time in the past.
     */
    RESUME_AT_CHANGE_TIME,
    /**
     * Set the startpoint using a user-defined Serializable object.
     */
    RESUME_AT_SERIALIZABLE,
    /**
     * Set the startpoint to the very beginning of the change log.
     */
    BEGINNING_OF_CHANGELOG,
    /**
     * Set the startpoint to the very end of the change log.
     */
    END_OF_CHANGELOG;
  }

  // the startpoint type
  private final StartpointType startpointType;

  // the possible startpoint values
  private String csnValue;
  private long changeNumber;
  private Date changeTime;
  private Serializable serializableValue;

  // the external server name, if one is required (change number, and change
  // time)
  private String serverName;

  /**
   * Constructor.
   * @param startpointType
   *          indicates the startpoint type (beginning of changelog, change
   *          time, etc)
   */
  private SetStartpointOptions(final StartpointType startpointType)
  {
    if(startpointType == null)
    {
      throw new NullPointerException(
              "The startpointType parameter cannot be null.");
    }
    this.startpointType = startpointType;
  }

  /**
   * Creates a SetStartpointOptions instance with the
   * <code>BEGINNING_OF_CHANGELOG</code> startpoint type.
   * @return a immutable instance of this class with only the startpoint type
   *         populated
   */
  public static SetStartpointOptions createBeginningOfChangelogOptions()
  {
    SetStartpointOptions options =
            new SetStartpointOptions(StartpointType.BEGINNING_OF_CHANGELOG);
    return options;
  }

  /**
   * Creates a SetStartpointOptions instance with the
   * <code>END_OF_CHANGELOG</code> startpoint type.
   * @return a immutable instance of this class with only the startpoint type
   *         populated
   */
  public static SetStartpointOptions createEndOfChangelogOptions()
  {
    SetStartpointOptions options =
            new SetStartpointOptions(StartpointType.END_OF_CHANGELOG);
    return options;
  }

  /**
   * Creates a SetStartpointOptions instance with the <code>RESUME_AT_CSN</code>
   * startpoint type.
   * @param csn
   *          the change sequence number of the first change to synchronize
   * @return a immutable instance of this class with the CSN value populated
   */
  public static SetStartpointOptions createResumeAtCSNOptions(final String csn)
  {
    if(csn == null)
    {
      throw new NullPointerException("The CSN parameter cannot be null");
    }
    SetStartpointOptions options =
            new SetStartpointOptions(StartpointType.RESUME_AT_CSN);
    options.csnValue = csn;
    return options;
  }

  /**
   * Creates a SetStartpointOptions instance with the
   * <code>RESUME_AT_CHANGE_NUMBER</code> startpoint type.
   * @param changeNumber
   *          the change number of the first change to synchronize
   * @param serverName
   *          the name of the External Server to which the change number
   *          applies. The CLI
   *          requires this parameter to be non-null, because a given change
   *          number may
   *          correspond to different changes on different replicas.
   * @return a immutable instance of this class with the change number and
   *         server name populated
   */
  public static SetStartpointOptions createResumeAtChangeNumberOptions(
          final long changeNumber, final String serverName)
  {
    if(serverName == null)
    {
      throw new NullPointerException("The serverName parameter cannot be null");
    }
    SetStartpointOptions options =
            new SetStartpointOptions(StartpointType.RESUME_AT_CHANGE_NUMBER);
    options.changeNumber = changeNumber;
    options.serverName = serverName;
    return options;
  }

  /**
   * Creates a SetStartpointOptions instance with the
   * <code>RESUME_AT_CHANGE_TIME</code> startpoint type. Changes
   * made prior to the specified change time will be ignored.
   * @param changeTime
   *          the cut-off time at which synchronization will resume
   * @param serverName
   *          the name of the External Server to which the change time applies.
   *          The CLI allows
   *          this parameter to be null, but some underlying sync source
   *          implementations
   *          may require it (such as DSEESyncSource).
   * @return a immutable instance of this class with the change time and server
   *         name populated
   */
  public static SetStartpointOptions createResumeAtChangeTimeOptions(
          final Date changeTime, final String serverName)
  {
    if(changeTime == null)
    {
      throw new NullPointerException("The changeTime parameter cannot be null");
    }
    SetStartpointOptions options =
            new SetStartpointOptions(StartpointType.RESUME_AT_CHANGE_TIME);
    options.changeTime = changeTime;
    options.serverName = serverName;
    return options;
  }

  /**
   * Creates a SetStartpointOptions instance with the
   * <code>RESUME_AT_SERIALIZABLE</code> startpoint type. Changes made prior to
   * the specified state will be ignored.
   * @param startpointValue
   *          a Serializable object which can be passed to the sync source as a
   *          valid
   *          indicator of where to start synchronization
   * @return a immutable instance of this class with the generic startpoint
   *         value populated
   */
  public static SetStartpointOptions createResumeAtSerializableOptions(
          final Serializable startpointValue)
  {
    SetStartpointOptions options =
            new SetStartpointOptions(StartpointType.RESUME_AT_SERIALIZABLE);
    options.serializableValue = startpointValue;
    return options;
  }

  /**
   * Gets the startpointType (i.e. RESUME_AT_CHANGE_NUMBER, END_OF_CHANGELOG,
   * etc)
   * @return a {@link StartpointType} instance
   */
  public StartpointType getStartpointType()
  {
    return this.startpointType;
  }

  /**
   * Gets the CSN value if one was specified.
   * @return a change sequence number, or null if the startpoint type is not
   *         <code>RESUME_AT_CSN</code>
   */
  public String getCSN()
  {
    return this.csnValue;
  }

  /**
   * Gets the change number value if one was specified.
   * @return a change number, or null if the startpoint type is not
   *         <code>RESUME_AT_CHANGE_NUMBER</code>
   */
  public long getChangeNumber()
  {
    return this.changeNumber;
  }

  /**
   * Gets the change time value if one was specified.
   * @return a {@link Date}, or null if the startpoint type is not
   *         <code>RESUME_AT_CHANGE_TIME</code>
   */
  public Date getChangeTime()
  {
    return this.changeTime;
  }

  /**
   * Gets the generic (serializable) startpoint value if one was specified.
   * @return a {@link Serializable} instance, or null if the startpoint type is
   *         not <code>RESUME_AT_SERIALIZABLE</code>
   */
  public Serializable getSerializableValue()
  {
    return this.serializableValue;
  }

  /**
   * Gets the server name if one was specified, representing the External Server
   * at which to
   * set the start point.
   * @return an External Server name, or null if one was not specified
   */
  public String getServerName()
  {
    return this.serverName;
  }
}
