/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2011-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.types;

import java.util.Collection;
import java.util.List;

/**
 * This interface represents the matched Sync Class for a {@link SyncOperation}.
 * While several Sync Classes may be defined for a Sync Pipe, an operation will
 * only match at most one of them. This can be used to obtain some of the
 * configured values for a particular Sync Class.
 */
public interface SyncClass
{
  /**
   * Returns the name of this Sync Class.
   *
   * @return a String representing the name of this Sync Class. This will never
   *         be null.
   */
  String getName();


  /**
   * Returns the configured description for this Sync Class.
   *
   * @return a String representing the description for this Sync Class. This
   *         may be null if no description is configured.
   */
  String getDescription();


  /**
   * Returns the set(s) of destination correlation attributes that are
   * configured on this Sync Class. When multiple attribute sets are configured,
   * the server iterates through them and uses the
   * first one that matches a single entry. Note that if the
   * <i>destination-correlation-attributes-on-delete</i> property
   * is set, then only those attributes will be used when correlating deleted
   * entries.
   *
   * @return a List of attribute sets. This will never be null.
   */
  List<Collection<String>> getDestinationCorrelationAttributes();


  /**
   * Returns the set(s) of destination correlation attributes that are
   * configured on this Sync Class for DELETE operations. When multiple
   * attribute sets are configured, the server iterates through them and uses
   * the first one that matches a single entry. Note that if the
   * <i>destination-correlation-attributes-on-delete</i> property
   * is not set, then the standard destination correlation attributes will be
   * used to correlate deleted entries.
   *
   * @return a List of attribute sets. This will never be null.
   */
  List<Collection<String>> getDestinationCorrelationAttributesOnDelete();


  /**
   * Returns the set of attributes that are never allowed to be modified once
   * an entry is created.
   *
   * @return a Collection of attribute names. This will never be null.
   */
  Collection<String> getDestinationCreateOnlyAttributes();


  /**
   * Returns whether this Sync Class will prevent writes to the destination that
   * contain empty (zero-length) values. Some servers do not allow this
   * behavior.
   *
   * @return true if this Sync Class is configured to ignore zero-length values
   *         (e.g. prevent them from being written), or false otherwise.
   */
  boolean isIgnoreZeroLengthValues();


  /**
   * Returns whether this Sync Class will use the ADD/DELETE modification type
   * or the REPLACE modification type when making modifications on the
   * destination.
   *
   * @return true if the ADD/DELETE modification types will be use, false if the
   *         REPLACE modification type will be used.
   */
  boolean isUseReversibleForm();


  /**
   * Returns whether this Sync class will convert modify operations into create
   * operations if the destination entry to be modified does not exist.
   *
   * @return true if the Sync Class will allow modifies to be converted to
   *         creates, false otherwise.
   */
  boolean isModifiesAsCreates();


  /**
   * Returns whether this Sync class will convert create operations into modify
   * operations if the destination entry to be modified already exists.
   *
   * @return true if the Sync Class will allow creates to be converted to
   *         modifies, false otherwise.
   */
  boolean isCreatesAsModifies();

}
