/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.types;

import java.util.Collection;
import java.util.List;

import com.unboundid.ldap.sdk.ChangeLogEntry;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.ldap.sdk.unboundidds.UnboundIDChangeLogEntry;

/**
 * This interface represents a single synchronized change from the Sync
 * Source to the Sync Destination.  It exists from when a change is detected
 * all the way through when the change is applied at the destination.
 */
public interface SyncOperation
{
  /**
   * Return the type of this SynchronizationOperation.
   *
   * @return  The type of this SynchronizationOperation.
   */
  SyncOperationType getType();



  /**
   * Return a String that can be used to correlate this change back to the
   * source entry. For example, if the source endpoint is a directory server,
   * this will be a DN.
   *
   * @return  A String that can be used to correlate this change back to the
   *          source entry.
   */
  String getIdentifiableInfo();



  /**
   * Return a unique ID for this {@code SyncOperation}. This ID is used in
   * log messages.
   *
   * @return  A unique ID for this operation.
   */
  String getUniqueId();



  /**
   * Return the Entry that was fetched from the sync source. This may be null
   * if the source entry no longer exists or has not yet been fetched.
   *
   * @return  The entry that was fetched from the source server.
   */
  Entry getSourceEntry();



  /**
   * Sets the entry that was fetched from the sync source.  If the entry was
   * deleted, then sufficient information about the entry must be provided so
   * that it can be correlated to a destination entry.
   *
   * @param  sourceEntry     The entry that was fetched from the source.
   * @param  isDeletedEntry  {@code true} if and only if the entry was deleted
   *                         at the source.
   */
  void setSourceEntry(Entry sourceEntry, boolean isDeletedEntry);



  /**
   * Return the original changelog entry that was detected for this operation.
   * This is only applicable when synchronizing from a data source that
   * supplies an LDAP changelog as described in
   * <i>draft-good-ldap-changelog</i>, such as the
   * ${DIRECTORY_SERVER_PRODUCT_NAME} or Sun Directory Server.
   * <p>
   * Note that this is the raw changelog entry and does not have any mappings
   * or filterings applied to it.
   *
   * @return a {@link ChangeLogEntry} instance, or null if the source endpoint
   *         does not use an LDAP changelog for change detection.
   */
  ChangeLogEntry getChangeLogEntry();



  /**
   * Return the original changelog entry that was detected for this operation.
   * This is only applicable when synchronizing from
   * ${A_DIRECTORY_SERVER_PRODUCT_NAME}. The UnboundID LDAP Changelog provides
   * additional functionality not present in other directory servers.
   * <p>
   * Note that this is the raw changelog entry and does not have any mappings
   * or filterings applied to it.
   *
   * @return a {@link UnboundIDChangeLogEntry} instance, or null if the source
   *         endpoint is not ${A_DIRECTORY_SERVER_PRODUCT_NAME}.
   */
  UnboundIDChangeLogEntry getUnboundIDChangeLogEntry();



  /**
   * Return the original change record that was detected for this operation.
   * This is only applicable when synchronizing from a data source that
   * is implemented by the <code>SyncSource</code> or
   * <code>ScriptedSyncSource</code> extension type.
   *
   * @return a {@link ChangeRecord} instance, or null if the source endpoint
   *         is not an instance of <code>SyncSource</code> or
   *         <code>ScriptedSyncSource</code>.
   */
  ChangeRecord getChangeRecord();



  /**
   * Return the DatabaseChangeRecord that was detected from a RDBMS. This is
   * only applicable when synchronizing from a database environment, and will
   * be null otherwise.
   *
   * @return a DatabaseChangeRecord that was detected from a database, or null
   *         if the sync source is not a JDBC sync source.
   */
  DatabaseChangeRecord getDatabaseChangeRecord();



  /**
   * Determines whether this SyncOperation represents a modify DN operation on
   * the source entry (if the source is an LDAP directory). The
   * result of this method is undefined if the source is a relational database
   * or some other type of endpoint.
   *
   * @return true if the entry has been renamed, false otherwise.
   */
  boolean isModifyDN();



  /**
   * Returns a representation of the target destination entry before it is
   * modified or renamed. This can be used to obtain the before-values of any
   * attributes. The returned entry cannot be modified.
   * <p>
   * This value is only defined after the SyncOperation has made it past the
   * post-mapping stage in the SyncPipe.
   *
   * @return a {@link Entry} instance.
   */
  Entry getDestinationEntryBeforeChange();



  /**
   * Returns a representation of the target destination entry after it is
   * modified or renamed. This can be used to obtain the after-values of any
   * attributes.
   * <p>
   * This value is only defined after the SyncOperation has made it past the
   * post-mapping stage in the SyncPipe.
   * <p>
   * The returned entry is modifiable and can be used to influence
   * the final list of modifications to apply at the destination. The
   * ${SYNC_SERVER_BASE_NAME} computes a diff between this entry and the
   * existing entry returned from the destination. Any modifications to this
   * entry would have to take place in either
   * {@code LDAPSyncDestinationPlugin.preFetch} or
   * {@code LDAPSyncDestinationPlugin.postFetch}.
   *
   * @return a {@link Entry} instance.
   */
  Entry getDestinationEntryAfterChange();



  /**
   * Return the names of the attributes that were modified at the source.
   *
   * @return  The names of the attributes that were modified at the source.
   */
  Collection<String> getModifiedSourceAttributes();



  /**
   * Add an additional source attribute that should be synchronized to the
   * destination.
   *
   * @param  attributeName  The attribute to add to the set of attributes to
   *                        synchronize.
   */
  void addModifiedSourceAttribute(String attributeName);



  /**
   * Remove a source attribute that should not be synchronized to the
   * destination.
   *
   * @param  attributeName  The attribute to remove from the set of attributes
   *                        to synchronize.
   */
  void removeModifiedSourceAttribute(String attributeName);



  /**
   * Add the name of a destination attribute that should be synchronized.
   *
   * @param  attributeName  The name of the destination attribute to
   *                        synchronize.
   */
  void addModifiedDestinationAttribute(String attributeName);



  /**
   * Return the name of the Sync Pipe for the operation being processed.
   *
   * @return  The name of the Sync Pipe for the operation being processed.
   */
  String getSyncPipeName();



  /**
   * Return the name of the Sync Class for the operation being processed.
   *
   * @return  The name of the Sync Class for the operation being processed.
   */
  String getSyncClassName();



  /**
   * Return the {@link SyncClass} for the operation being processed. This can
   * be used to determine the correlation attributes and other properties
   * configured for this type of change.
   *
   * @return  The Sync Class that this operation falls into, or null if no
   *          Sync Class has been determined yet.
   */
  SyncClass getSyncClass();



  /**
   * Stores an attachment in the operation.  This can be used by an extension
   * to have information carried in the synchronization operation from one
   * extension point to the next.
   *
   * @param  key    The key of the attachment.  To avoid conflicts, the
   *                fully-qualified class name of the extension should be used
   *                as a prefix of the key.
   *
   * @param  value  The value of the attachment.
   */
  void putAttachment(Object key, Object value);



  /**
   * Returns an attachment from the operation that was previously set with a
   * call to {@link #putAttachment}.
   *
   * @param  key  The key of the attachment.
   *
   * @return  The value of the attachment or {@code null} if no attachment was
   *          set for the specified key.
   */
  Object getAttachment(Object key);



  /**
   * Returns a list of warning messages (if any) about this SyncOperation. For
   * example, if a changelog entry omitted some data because an attribute had
   * too many values, there will be a message in the returned list indicating
   * so. Typically synchronization of this SyncOperation can continue despite
   * these warnings.
   *
   * @return a list of warning messages; the list will be empty if there are no
   *         warnings.
   */
  List<String> getWarnings();



  /**
   * Logs an error message to the synchronization log for this change.  Messages
   * logged here should pertain to this specific change.  For general logging
   * capabilities, see {@link SyncServerContext}
   *
   * @param  message  The message to log.
   */
  void logError(String message);



  /**
   * Logs an informational message to the synchronization log for this change.
   * Messages logged here should pertain to this specific change.  For general
   * logging capabilities, see {@link SyncServerContext}
   *
   * @param  message  The message to log.
   */
  void logInfo(String message);



  /**
   * Logs a debug message to the synchronization log for this change. Messages
   * logged here should pertain to this specific change.  For general logging
   * capabilities, see {@link SyncServerContext}.  To see messages logged with
   * this API the Sync Log Publisher should include plugin-debug.  For resyncs
   * pass "--logLevel debug".
   *
   * @param  message  The message to log.
   */
  void logDebug(String message);



  /**
   * Get the "ignored" flag. For create, modify, or delete operations when the
   * "ignored" flag is set to true the Sync server does not include the
   * operation in the statistics listed under
   * "cn=Sync Pipe Monitor: &lt;pipe name&gt;,cn=monitor".
   *
   * @return The "ignored" flag.
   */
  boolean isIgnored();



  /**
   * Sets the "ignored" flag to true. Setting the "ignored" flag to true informs
   * the Sync server that the create, modify, or delete operation was ignored.
   * This prevents the operation from being included as an applied operation in
   * the statistics that are reported for the Sync Pipe via the Management
   * Console, the bin/status command line utility, and in a raw form under
   * "cn=Sync Pipe Monitor: &lt;pipe name&gt;,cn=monitor".
   */
  void setIgnored();
}
