/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.types;

import static com.unboundid.directory.sdk.sync.util.ScriptUtils
  .addBinaryAttribute;
import static com.unboundid.directory.sdk.sync.util.ScriptUtils
  .addBooleanAttribute;
import static com.unboundid.directory.sdk.sync.util.ScriptUtils
  .addDateAttribute;
import static com.unboundid.directory.sdk.sync.util.ScriptUtils
  .addNumericAttribute;
import static com.unboundid.directory.sdk.sync.util.ScriptUtils
  .addStringAttribute;

import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.util.StaticUtils;

/**
 * This class wraps an open JDBC Connection and provides controlled access to it
 * via a handful of delegating methods. The underlying connection is always part
 * of a new transaction. The ${SYNC_SERVER_BASE_NAME} will always automatically
 * commit or rollback the transaction after a method is finished with it,
 * depending on if the method returned successfully or threw an exception.
 * <p>
 * There are also facilities for setting a timeout, after
 * which the TransactionContext will automatically become invalidated, and
 * subsequent operations on it will throw a {@link RuntimeException}.
 * </p>
 */
public final class TransactionContext
{

  // The underlying JDBC Connection (this is actually a DBCP delegating
  // connection)
  private final Connection connection;

  // Timer used to invalidate TransactionContexts
  private static final Timer timer = new Timer(true);

  // The timer task local to this instance
  private TimerTask timeOutTask;

  // The number of millis after which to timeout
  private long timeOutMs;

  // Whether this context has timed out
  private volatile boolean isTimedOut = false;

  // Whether this context has been closed
  private volatile boolean isClosed = false;

  /**
   * Constructor. The connection instance may not be null.
   * @param connection
   *          a fresh JDBC Connection from the pool
   */
  public TransactionContext(final Connection connection)
  {
    if(connection == null)
    {
      throw new NullPointerException("Connection is null.");
    }
    this.connection = connection;

    try
    {
      // Start a fresh transaction
      this.connection.commit();
    }
    catch(SQLException e)
    {
      throw new IllegalStateException(
              "Could not initialize TransactionContext:" +
                                  StaticUtils.getExceptionMessage(e));
    }
  }

  /**
   * Executes the given SQL query and converts the results into a raw LDAP
   * {@link Entry}, where the attribute names are the raw column names from the
   * result of the query. The <code>rdnColumn</code> parameter is the column
   * name
   * to use as the RDN attribute for the entry. For example if the rdnColumn is
   * "accountID", then the DN of the returned Entry will be
   * "accountID={accountID}".
   * @param sql
   *          the SQL query to execute
   * @param rdnColumn
   *          the column name in the result set which is to be used for
   *          the DN of the returned {@link Entry}
   * @return the resulting LDAP Entry
   * @throws SQLException
   *           if a database access error occurs or the query matches more than
   *           one row
   */
  public synchronized Entry searchToRawEntry(final String sql,
          final String rdnColumn)
          throws SQLException
  {
    checkStatus();
    PreparedStatement stmt = connection.prepareStatement(sql);
    return searchToRawEntry(stmt, rdnColumn);
  }

  /**
   * Executes the given {@link PreparedStatement} and converts the results into
   * a raw LDAP {@link Entry}, where the attribute names are the raw column
   * names from the
   * result of the query. The <code>rdnColumn</code> parameter is the column
   * name
   * to use as the RDN attribute for the entry. For example if the rdnColumn is
   * "accountID", then the DN of the returned Entry will be
   * "accountID={accountID}".
   * @param statement
   *          the PreparedStatement to execute
   * @param rdnColumn
   *          the column name in the result set which is to be used for
   *          the DN of the returned {@link Entry}
   * @return the resulting LDAP Entry
   * @throws SQLException
   *           if a database access error occurs or the query matches more than
   *           one row
   */
  public synchronized Entry searchToRawEntry(final PreparedStatement statement,
          final String rdnColumn) throws SQLException
  {
    checkStatus();
    ResultSet rset = statement.executeQuery();
    Entry entry = null;
    try
    {
      if(rset.next())
      {
        Object rdnValue = rset.getObject(rdnColumn);
        if(rdnValue == null)
        {
          throw new SQLException("The RDN column '" + rdnColumn +
                  "' was not found in the resulting row.");
        }
        entry = new Entry(rdnColumn + "=" + rdnValue);
        ResultSetMetaData metaData = rset.getMetaData();
        for(int i = 1; i <= metaData.getColumnCount(); i++)
        {
          Object o = rset.getObject(i);
          String attrName = metaData.getColumnLabel(i);
          String columnClass = metaData.getColumnClassName(i);

          // figure out what type of class this column maps to
          Class<?> clazz = null;
          try
          {
            clazz = Class.forName(columnClass);
          }
          catch(ClassNotFoundException e)
          {
            throw new SQLException("Couldn't create class for column: " +
                    attrName, e);
          }

          if(String.class.isAssignableFrom(clazz))
          {
            String str = (String) o;
            addStringAttribute(entry, attrName, str);
          }
          else if(Number.class.isAssignableFrom(clazz))
          {
            Number num = (Number) o;
            addNumericAttribute(entry, attrName, num);
          }
          else if(Date.class.isAssignableFrom(clazz))
          {
            Date date = (Date) o;
            addDateAttribute(entry, attrName, date, true);
          }
          else if(Character.class.isAssignableFrom(clazz))
          {
            Character c = (Character) o;
            addStringAttribute(entry, attrName, c.toString());
          }
          else if(Boolean.class.isAssignableFrom(clazz))
          {
            Boolean b = (Boolean) o;
            addBooleanAttribute(entry, attrName, b);
          }
          else if(Blob.class.isAssignableFrom(clazz))
          {
            Blob blob = (Blob) o;
            addBinaryAttribute(entry, attrName, blob, Integer.MAX_VALUE);
          }
          else if(Clob.class.isAssignableFrom(clazz))
          {
            Clob clob = (Clob) o;
            if(clob != null)
            {
              addStringAttribute(entry, attrName,
                      clob.getSubString(1, Integer.MAX_VALUE));
              clob.free();
            }
          }
          else
          {
            throw new SQLException("Column '" + attrName +
                    "' has an unhandled type: " + columnClass);
          }
        }
      }
      if(rset.next())
      {
        throw new SQLException("The query matched more than one row.");
      }
    }
    finally
    {
      rset.close();
      statement.close();
    }
    return entry;
  }

  /**
   * Creates a <code>Statement</code> object for sending
   * SQL statements to the database.
   * SQL statements without parameters are normally
   * executed using <code>Statement</code> objects. If the same SQL statement
   * is executed many times, it may be more efficient to use a
   * <code>PreparedStatement</code> object.
   * <P>
   * Result sets created using the returned <code>Statement</code> object will
   * by default be type <code>TYPE_FORWARD_ONLY</code> and have a concurrency
   * level of <code>CONCUR_READ_ONLY</code>. The holdability of the created
   * result sets can be determined by calling {@link Connection#getHoldability}.
   * @return a new default <code>Statement</code> object
   * @throws SQLException
   *           if a database access error occurs
   *           or this method is called on a closed connection
   */
  public synchronized Statement createStatement() throws SQLException
  {
    checkStatus();
    return connection.createStatement();
  }

  /**
   * Creates a <code>PreparedStatement</code> object for sending
   * parameterized SQL statements to the database.
   * <P>
   * A SQL statement with or without IN parameters can be pre-compiled and
   * stored in a <code>PreparedStatement</code> object. This object can then be
   * used to efficiently execute this statement multiple times.
   * <P>
   * <B>Note:</B> This method is optimized for handling parametric SQL
   * statements that benefit from precompilation. If the driver supports
   * precompilation, the method <code>prepareStatement</code> will send the
   * statement to the database for precompilation. Some drivers may not support
   * precompilation. In this case, the statement may not be sent to the database
   * until the <code>PreparedStatement</code> object is executed. This has no
   * direct effect on users; however, it does affect which methods throw certain
   * <code>SQLException</code> objects.
   * <P>
   * Result sets created using the returned <code>PreparedStatement</code>
   * object will by default be type <code>TYPE_FORWARD_ONLY</code> and have a
   * concurrency level of <code>CONCUR_READ_ONLY</code>. The holdability of the
   * created result sets can be determined by calling
   * {@link Connection#getHoldability}.
   * @param sql
   *          an SQL statement that may contain one or more '?' IN
   *          parameter placeholders
   * @return a new default <code>PreparedStatement</code> object containing the
   *         pre-compiled SQL statement
   * @throws SQLException
   *           if a database access error occurs
   *           or this method is called on a closed connection
   */
  public synchronized PreparedStatement prepareStatement(final String sql)
          throws SQLException
  {
    checkStatus();
    return connection.prepareStatement(sql);
  }

  /**
   * Creates a <code>CallableStatement</code> object for calling
   * database stored procedures.
   * The <code>CallableStatement</code> object provides
   * methods for setting up its IN and OUT parameters, and
   * methods for executing the call to a stored procedure.
   * <P>
   * <B>Note:</B> This method is optimized for handling stored procedure call
   * statements. Some drivers may send the call statement to the database when
   * the method <code>prepareCall</code> is done; others may wait until the
   * <code>CallableStatement</code> object is executed. This has no direct
   * effect on users; however, it does affect which method throws certain
   * SQLExceptions.
   * <P>
   * Result sets created using the returned <code>CallableStatement</code>
   * object will by default be type <code>TYPE_FORWARD_ONLY</code> and have a
   * concurrency level of <code>CONCUR_READ_ONLY</code>. The holdability of the
   * created result sets can be determined by calling
   * {@link Connection#getHoldability}.
   * @param sql
   *          an SQL statement that may contain one or more '?'
   *          parameter placeholders. Typically this statement is specified
   *          using JDBC
   *          call escape syntax.
   * @return a new default <code>CallableStatement</code> object containing the
   *         pre-compiled SQL statement
   * @throws SQLException
   *           if a database access error occurs
   *           or this method is called on a closed connection
   */
  public synchronized CallableStatement prepareCall(final String sql)
          throws SQLException
  {
    checkStatus();
    return connection.prepareCall(sql);
  }

  /**
   * Retrieves the current transaction isolation level for the underlying
   * <code>Connection</code>.
   * @return the current transaction isolation level, which will be one
   *         of the following constants:
   *         <code>Connection.TRANSACTION_READ_UNCOMMITTED</code>,
   *         <code>Connection.TRANSACTION_READ_COMMITTED</code>,
   *         <code>Connection.TRANSACTION_REPEATABLE_READ</code>,
   *         <code>Connection.TRANSACTION_SERIALIZABLE</code>, or
   *         <code>Connection.TRANSACTION_NONE</code>.
   * @throws SQLException
   *           if a database access error occurs or this method is called on a
   *           closed connection
   */
  public synchronized int getTransactionIsolation() throws SQLException
  {
    checkStatus();
    return connection.getTransactionIsolation();
  }

  /**
   * Attempts to change the transaction isolation level for the underlying
   * <code>Connection</code> object to the one given.
   * The constants defined in the interface {@link Connection} are the possible
   * transaction isolation levels.
   * <P>
   * <B>Note:</B> If this method is called during a transaction, the result is
   * implementation-defined.
   * @param level
   *          one of the following <code>Connection</code> constants:
   *          <code>Connection.TRANSACTION_READ_UNCOMMITTED</code>,
   *          <code>Connection.TRANSACTION_READ_COMMITTED</code>,
   *          <code>Connection.TRANSACTION_REPEATABLE_READ</code>, or
   *          <code>Connection.TRANSACTION_SERIALIZABLE</code>.
   *          (Note that <code>Connection.TRANSACTION_NONE</code> cannot be used
   *          because it specifies that transactions are not supported.)
   * @throws SQLException
   *           if a database access error occurs, this
   *           method is called on a closed connection
   *           or the given parameter is not one of the <code>Connection</code>
   *           constants
   */
  public synchronized void setTransactionIsolation(final int level)
          throws SQLException
  {
    checkStatus();
    connection.setTransactionIsolation(level);
  }

  /**
   * Commits the internal Connection if the transaction has not timed out and is
   * still valid.
   * @throws SQLException if a database access error occurs
   */
  public synchronized void commit() throws SQLException
  {
    checkStatus();
    connection.commit();
  }

  /**
   * Rolls back the internal Connection if the transaction has not timed out and
   * is still valid.
   * @throws SQLException if a database access error occurs
   */
  public synchronized void rollBack() throws SQLException
  {
    checkStatus();
    connection.rollback();
  }

  /**
   * Releases the internal connection back the pool and invalidates this context
   * object.
   */
  public synchronized void close()
  {
    if(timeOutTask != null)
    {
      timeOutTask.cancel();
    }
    try
    {
      connection.close(); // return the connection to the pool
    }
    catch(SQLException e)
    {
      //suppress
    }
    finally
    {
      isClosed = true;
    }
  }

  /**
   * Gets the underlying {@link Connection} instance used by this context. This
   * is not assignable to any vendor-specific types (e.g. OracleConnection or
   * SQLServerConnection).
   * If possible, you should avoid getting the connection directly, and instead
   * use the
   * wrapper methods provided by this class.
   * @return a valid Connection instance
   */
  synchronized Connection getConnection()
  {
    checkStatus();
    return connection;
  }

  /**
   * Sets a timeout for this TransactionContext. The count begins immediately
   * when this method is called. This method may be called multiple times if the
   * timeout needs to be reset, provided that a previous timeout hasn't already
   * expired. A value of zero indicates that there should be no timeout.
   * Negative values are not allowed.
   * <p>
   * After the timeout has expired, all methods except {@link #isTimedOut} and
   * {@link #isClosed} will throw a <code>IllegalStateException</code>.
   * @param timeOutMillis
   *          the delay in milliseconds after which this TransactionContext will
   *          time out
   */
  public synchronized void setTimeout(final long timeOutMillis)
  {
    checkStatus();
    isTimedOut = false;
    timeOutMs = timeOutMillis;
    if(timeOutTask != null)
    {
      timeOutTask.cancel();
    }
    timer.purge();
    if(timeOutMillis == 0)
    {
      return;
    }
    timeOutTask = new TimerTask()
    {
      @Override
      public void run()
      {
        try
        {
          rollBack();
        }
        catch (Throwable t)
        {
          //suppress
        }
        close();
        isTimedOut = true;
      }
    };
    timer.schedule(timeOutTask, timeOutMillis);
  }

  /**
   * Checks whether this TransactionContext has timed out.
   * <p>
   * After the timeout has expired, all methods except {@link #isTimedOut},
   * {@link #isClosed}, and <code>log[Error|Info|Debug|Exception]</code> will
   * throw an <code>IllegalStateException</code>.
   * @return true if the timeout has expired, false if not.
   */
  public boolean isTimedOut()
  {
    return isTimedOut;
  }

  /**
   * Checks whether this TransactionContext is closed.
   * @return true if the context is closed and no longer usable, false if not.
   */
  public boolean isClosed()
  {
    return isClosed;
  }

  /**
   * Throws an IllegalStateException if this context is timed out or closed.
   */
  private void checkStatus()
  {
    if(isTimedOut)
    {
      throw new IllegalStateException("The transaction timed out after " +
              timeOutMs + "ms.");
    }
    else if(isClosed)
    {
      throw new IllegalStateException(
              "The transaction has already been closed.");
    }
  }
}
