/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * docs/licenses/cddl.txt
 * or http://www.opensource.org/licenses/cddl1.php.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * docs/licenses/cddl.txt.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2010-2019 Ping Identity Corporation
 */
package com.unboundid.directory.sdk.sync.util;

import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.ldap.sdk.RDN;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.sql.Blob;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.regex.Pattern;

/**
 * This class contains various utility methods for working with the
 * UnboundID LDAP SDK and JDBC objects within script implementations. These
 * are useful for populating LDAP entries with content from JDBC result sets
 * and also for working with DNs, Timestamps, and other objects.
 */
@ThreadSafety(level = ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class ScriptUtils
{

  /**
   * The date format string that will be used to construct and parse dates
   * represented using generalized time with a four-digit year.
   */
  private static final String DATE_FORMAT_GMT_TIME =
          "yyyyMMddHHmmss'Z'";

  /**
   * The date format string that will be used to construct and parse dates
   * represented using generalized time.
   */
  private static final String DATE_FORMAT_GENERALIZED_TIME =
          "yyyyMMddHHmmss.SSS'Z'";

  // DateFormats are not thread-safe, so we provide a thread local
  private static final ThreadLocal<SimpleDateFormat> gmtTime =
    new ThreadLocal<SimpleDateFormat>()
        {
          @Override
          protected SimpleDateFormat initialValue()
          {
            return new SimpleDateFormat(DATE_FORMAT_GMT_TIME);
          }
        };

  private static final ThreadLocal<SimpleDateFormat> generalizedTime =
    new ThreadLocal<SimpleDateFormat>()
        {
          @Override
          protected SimpleDateFormat initialValue()
          {
            return new SimpleDateFormat(DATE_FORMAT_GENERALIZED_TIME);
          }
        };

  // Precompiled regular expressions used to remove spaces.
  private static final Pattern leadingSpaceRE = Pattern.compile("^ *");
  private static final Pattern trailingSpaceRE = Pattern.compile(" *$");

  /**
   * Private constructor to enforce non-instantiability.
   */
  private ScriptUtils() {}

  /**
   * Adds an {@link Attribute} with the given attribute name and string value to
   * the given entry if the value is not null.
   * @param entry
   *          an LDAP entry instance. May be null.
   * @param attrName
   *          the name of the attribute to add to the entry. May <b>not</b> be
   *          null.
   * @param value
   *          the value for the attribute to add to the entry. May be null.
   */
  public static void addStringAttribute(final Entry entry,
                                        final String attrName,
                                        final String value)
  {
    if(entry != null && value != null)
    {
      entry.addAttribute(attrName, value);
    }
  }

  /**
   * Adds an {@link Attribute} with the given attribute name and numeric value
   * to the given entry if the value is not null.
   * @param entry
   *          an LDAP entry instance. May be null.
   * @param attrName
   *          the name of the attribute to add to the entry. May <b>not</b> be
   *          null.
   * @param value
   *          the value for the attribute to add to the entry. May be null.
   */
  public static void addNumericAttribute(final Entry entry,
                                         final String attrName,
                                         final Number value)
  {
    if(entry != null && value != null)
    {
      entry.addAttribute(attrName, value.toString());
    }
  }

  /**
   * Adds an {@link Attribute} with the given attribute name and boolean value
   * to the given entry if the value is not null. If the boolean is
   * <code>true</code>, the attribute value will be the string "true", otherwise
   * it will be the string "false".
   * @param entry
   *          an LDAP entry instance. May be null.
   * @param attrName
   *          the name of the attribute to add to the entry. May <b>not</b> be
   *          null.
   * @param value
   *          the value for the attribute to add to the entry. May be null.
   */
  public static void addBooleanAttribute(final Entry entry,
                                         final String attrName,
                                         final Boolean value)
  {
    if(entry != null && value != null)
    {
      entry.addAttribute(attrName, value.toString());
    }
  }

  /**
   * Adds an {@link Attribute} with the given attribute name and {@link Date}
   * value to the given entry if the value is not null. The date is formatted
   * using the generalized time syntax with a four-digit year and an optional
   * milliseconds component (i.e. yyyyMMddHHmmss[.SSS]'Z').
   * @param entry
   *          entry an LDAP entry instance. May be null.
   * @param attrName
   *          the name of the attribute to add to the entry. May <b>not</b> be
   *          null.
   * @param date
   *          the Date value for the attribute to add to the entry. May be null.
   * @param includeMilliseconds
   *          whether to include the milliseconds component in the attribute
   *          value
   */
  public static void addDateAttribute(final Entry entry,
                                      final String attrName,
                                      final Date date,
                                      final boolean includeMilliseconds)
  {
    if(entry != null && date != null)
    {
      String result;
      if(includeMilliseconds)
      {
        result = generalizedTime.get().format(date);
      }
      else
      {
        result = gmtTime.get().format(date);
      }
      entry.addAttribute(attrName, result);
    }
  }

  /**
   * Adds an {@link Attribute} with the given attribute name and {@link Blob}
   * value to the given entry if the underlying byte array is not null or empty.
   * This method calls <code>free()</code> on the Blob object after the
   * attribute has been added to the entry.
   * @param entry
   *          an LDAP entry instance. May be null.
   * @param attrName
   *          the name of the attribute to add to the entry. May <b>not</b> be
   *          null.
   * @param value
   *          the value for the attribute to add to the entry. May be null.
   * @param maxBytes
   *          the maximum number of bytes to extract from the Blob
   *          and add to the entry.
   */
  public static void addBinaryAttribute(final Entry entry,
                                        final String attrName,
                                        final Blob value,
                                        final int maxBytes)
  {
    if(entry != null && value != null)
    {
      try
      {
        byte[] bytes = value.getBytes(1, maxBytes);
        if(bytes.length > 0)
        {
          entry.addAttribute(attrName, bytes);
        }
        value.free();
      }
      catch(SQLException e)
      {
        // suppress
      }
    }
  }

  /**
   * Returns true if the given attribute is not null and contains any one or
   * more of the given string values; returns false otherwise. This method
   * is case-insensitive.
   * @param attr
   *          the {@link Attribute} whose values to check
   * @param values
   *          the value(s) you are looking for
   * @return true if any of the values were found in the attribute, false if not
   */
  public static boolean containsAnyValue(final Attribute attr,
                                         final String... values)
  {
    if(attr == null)
    {
      return false;
    }
    for(String attrValue : attr.getValues())
    {
      for(String valueToCheck : values)
      {
        if(attrValue.equalsIgnoreCase(valueToCheck))
        {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * String helper method to check if a value is a null object
   * or empty string.
   * @param value
   *          the String object to check
   * @return true if the value is null or empty string, false otherwise
   */
  public static boolean isNullOrEmpty(final String value)
  {
    if(value == null)
    {
      return true;
    }
    return value.isEmpty();
  }

  /**
   * Returns a SQL {@link Timestamp} based on the string value that is passed
   * in. The string is parsed using generalized time syntax first with and then
   * without milliseconds (i.e. yyyyMMddHHmmss[.SSS]'Z'). If the string cannot
   * be parsed, <code>null</code> is returned.
   * @param value
   *          a string that represents a timestamp in generalized time format
   * @return a SQL Timestamp value set to the time that was passed in, or null
   *         if the value cannot be parsed
   */
  public static Timestamp getTimestampFromString(final String value)
  {
    Timestamp ts = null;
    if(value == null)
    {
      return ts;
    }

    // generalized time (with milliseconds) is a more specific format,
    // so try to parse as that first
    try
    {
      Date d = generalizedTime.get().parse(value);
      ts = new Timestamp(d.getTime());
      return ts;
    }
    catch(ParseException e)
    {
    }

    // try to parse as GMT time
    try
    {
      Date d = gmtTime.get().parse(value);
      ts = new Timestamp(d.getTime());
    }
    catch(ParseException e)
    {
    }

    return ts;
  }

  /**
   * Takes an identifier string (for example from the database changelog table)
   * and creates a DN from the components. If there are multiple primary keys
   * in the identifier, they must be delimited by a unique string with which
   * the identifier can be split. For example, you could specify a delimiter of
   * "%%" to handle the following identifier: account_id=123%%group_id=5.
   * <p>
   * The resulting DN will contain a RDN per component, and the relative order
   * of RDNs will be consistent with the order of the components in the original
   * identifier string. The components here are usually
   * primary keys for the entry in the database.
   * @param identifiableInfo
   *          the identifier string for a given database entry. This cannot be
   *          null.
   * @param delimiter
   *          The delimiter used to split separate components of the
   *          identifiable info. If this is null, the default of "%%" will be
   *          used.
   * @return a DN representing the given identifier.
   */
  public static DN idStringToDN(final String identifiableInfo,
                                final String delimiter)
  {
    String defaultDelimiter = delimiter;
    if(delimiter == null || delimiter.isEmpty())
    {
      defaultDelimiter = "%%"; //default delimiter
    }

    List<RDN> rdnList = new ArrayList<RDN>();
    String[] pairs = identifiableInfo.split(defaultDelimiter);
    for(String pair : pairs)
    {
      String[] kv = pair.split("=", 2);
      if(kv.length != 2)
      {
        throw new IllegalArgumentException("Malformed identifier component: " +
                    pair);
      }

      // Strip of leading and trailing spaces.  This is like String.trim()
      // except that only spaces are removed.
      String key = trailingSpaceRE.matcher(
              leadingSpaceRE.matcher(kv[0]).replaceAll("")).replaceAll("");
      String value = trailingSpaceRE.matcher(
              leadingSpaceRE.matcher(kv[1]).replaceAll("")).replaceAll("");

      rdnList.add(new RDN(key, value));
    }

    if(rdnList.isEmpty())
    {
      throw new IllegalArgumentException(
              "The identifiableInfo parameter is empty.");
    }

    return new DN(rdnList);
  }

  /**
   * Takes an identifier DN (such as the output from
   * {@link #idStringToDN(String,String)} and
   * creates a hash map of the components (RDN attributes) to their respective
   * values. Each RDN will have a separate entry in the resulting map.
   * <p>
   * This method is meant to handle database entry identifier DNs, and as such
   * does not handle DNs with multi-valued RDNs (i.e.
   * pk1=John+pk2=Doe,groupID=123).
   * @param identifiableInfo
   *          the identifier DN for a particular database entry. This cannot be
   *          null.
   * @return a map of each RDN name to its value (from the given DN)
   */
  public static Map<String, String> dnToMap(final DN identifiableInfo)
  {

    Map<String, String> ids = new HashMap<String, String>();
    for(RDN rdn : identifiableInfo.getRDNs())
    {
      String[] kv = rdn.toString().split("=", 2);
      ids.put(kv[0], kv[1]);
    }
    return ids;
  }
}
