/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.ldap.sdk;

import com.unboundid.ldap.sdk.AbstractConnectionPool;
import com.unboundid.ldap.sdk.BindRequest;
import com.unboundid.ldap.sdk.DisconnectType;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionOptions;
import com.unboundid.ldap.sdk.LDAPConnectionPoolHealthCheck;
import com.unboundid.ldap.sdk.LDAPConnectionPoolHealthCheckThread;
import com.unboundid.ldap.sdk.LDAPConnectionPoolStatistics;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPMessages;
import com.unboundid.ldap.sdk.PostConnectProcessor;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.ServerSet;
import com.unboundid.ldap.sdk.SingleServerSet;
import com.unboundid.util.Debug;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.Validator;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class LDAPThreadLocalConnectionPool
extends AbstractConnectionPool {
    private static final long DEFAULT_HEALTH_CHECK_INTERVAL = 60000L;
    private volatile boolean closed;
    private final BindRequest bindRequest;
    private final ConcurrentHashMap<Thread, LDAPConnection> connections;
    private LDAPConnectionPoolHealthCheck healthCheck;
    private final LDAPConnectionPoolHealthCheckThread healthCheckThread;
    private final LDAPConnectionPoolStatistics poolStatistics;
    private volatile long healthCheckInterval;
    private volatile long maxConnectionAge;
    private final PostConnectProcessor postConnectProcessor;
    private final ServerSet serverSet;
    private String connectionPoolName;

    public LDAPThreadLocalConnectionPool(LDAPConnection connection) throws LDAPException {
        this(connection, null);
    }

    public LDAPThreadLocalConnectionPool(LDAPConnection connection, PostConnectProcessor postConnectProcessor) throws LDAPException {
        Validator.ensureNotNull(connection);
        this.postConnectProcessor = postConnectProcessor;
        this.healthCheck = new LDAPConnectionPoolHealthCheck();
        this.healthCheckInterval = 60000L;
        this.poolStatistics = new LDAPConnectionPoolStatistics(this);
        this.connectionPoolName = null;
        if (!connection.isConnected()) {
            throw new LDAPException(ResultCode.PARAM_ERROR, LDAPMessages.ERR_POOL_CONN_NOT_ESTABLISHED.get());
        }
        this.serverSet = new SingleServerSet(connection.getConnectedAddress(), connection.getConnectedPort(), connection.getLastUsedSocketFactory(), connection.getConnectionOptions());
        this.bindRequest = connection.getLastBindRequest();
        this.connections = new ConcurrentHashMap();
        this.connections.put(Thread.currentThread(), connection);
        this.maxConnectionAge = 0L;
        this.closed = false;
        this.healthCheckThread = new LDAPConnectionPoolHealthCheckThread(this);
        this.healthCheckThread.start();
    }

    public LDAPThreadLocalConnectionPool(ServerSet serverSet, BindRequest bindRequest) {
        this(serverSet, bindRequest, null);
    }

    public LDAPThreadLocalConnectionPool(ServerSet serverSet, BindRequest bindRequest, PostConnectProcessor postConnectProcessor) {
        Validator.ensureNotNull(serverSet);
        this.serverSet = serverSet;
        this.bindRequest = bindRequest;
        this.postConnectProcessor = postConnectProcessor;
        this.healthCheck = new LDAPConnectionPoolHealthCheck();
        this.healthCheckInterval = 60000L;
        this.poolStatistics = new LDAPConnectionPoolStatistics(this);
        this.connectionPoolName = null;
        this.connections = new ConcurrentHashMap();
        this.maxConnectionAge = 0L;
        this.closed = false;
        this.healthCheckThread = new LDAPConnectionPoolHealthCheckThread(this);
        this.healthCheckThread.start();
    }

    private LDAPConnection createConnection() throws LDAPException {
        LDAPConnection c = this.serverSet.getConnection(this.healthCheck);
        c.setConnectionPool(this);
        LDAPConnectionOptions opts = c.getConnectionOptions();
        if (opts.autoReconnect()) {
            opts = opts.duplicate();
            opts.setAutoReconnect(false);
            c.setConnectionOptions(opts);
        }
        if (this.postConnectProcessor != null) {
            try {
                this.postConnectProcessor.processPreAuthenticatedConnection(c);
            }
            catch (Exception e) {
                Debug.debugException(e);
                try {
                    this.poolStatistics.incrementNumFailedConnectionAttempts();
                    c.setDisconnectInfo(DisconnectType.POOL_CREATION_FAILURE, null, e);
                    c.terminate(null);
                }
                catch (Exception e2) {
                    Debug.debugException(e2);
                }
                if (e instanceof LDAPException) {
                    throw (LDAPException)e;
                }
                throw new LDAPException(ResultCode.CONNECT_ERROR, LDAPMessages.ERR_POOL_POST_CONNECT_ERROR.get(StaticUtils.getExceptionMessage(e)), e);
            }
        }
        try {
            if (this.bindRequest != null) {
                c.bind(this.bindRequest.duplicate());
            }
        }
        catch (Exception e) {
            Debug.debugException(e);
            try {
                this.poolStatistics.incrementNumFailedConnectionAttempts();
                c.setDisconnectInfo(DisconnectType.BIND_FAILED, null, e);
                c.terminate(null);
            }
            catch (Exception e2) {
                Debug.debugException(e2);
            }
            if (e instanceof LDAPException) {
                throw (LDAPException)e;
            }
            throw new LDAPException(ResultCode.CONNECT_ERROR, LDAPMessages.ERR_POOL_CONNECT_ERROR.get(StaticUtils.getExceptionMessage(e)), e);
        }
        if (this.postConnectProcessor != null) {
            try {
                this.postConnectProcessor.processPostAuthenticatedConnection(c);
            }
            catch (Exception e) {
                Debug.debugException(e);
                try {
                    this.poolStatistics.incrementNumFailedConnectionAttempts();
                    c.setDisconnectInfo(DisconnectType.POOL_CREATION_FAILURE, null, e);
                    c.terminate(null);
                }
                catch (Exception e2) {
                    Debug.debugException(e2);
                }
                if (e instanceof LDAPException) {
                    throw (LDAPException)e;
                }
                throw new LDAPException(ResultCode.CONNECT_ERROR, LDAPMessages.ERR_POOL_POST_CONNECT_ERROR.get(StaticUtils.getExceptionMessage(e)), e);
            }
        }
        c.setConnectionPoolName(this.connectionPoolName);
        this.poolStatistics.incrementNumSuccessfulConnectionAttempts();
        return c;
    }

    public void close() {
        this.closed = true;
        this.healthCheckThread.stopRunning();
        Iterator<Map.Entry<Thread, LDAPConnection>> iterator = this.connections.entrySet().iterator();
        while (iterator.hasNext()) {
            LDAPConnection conn = iterator.next().getValue();
            iterator.remove();
            conn.setDisconnectInfo(DisconnectType.POOL_CLOSED, null, null);
            conn.terminate(null);
        }
    }

    public LDAPConnection getConnection() throws LDAPException {
        Thread t = Thread.currentThread();
        LDAPConnection conn = this.connections.get(t);
        if (this.closed) {
            if (conn != null) {
                conn.terminate(null);
                this.connections.remove(t);
            }
            this.poolStatistics.incrementNumFailedCheckouts();
            throw new LDAPException(ResultCode.CONNECT_ERROR, LDAPMessages.ERR_POOL_CLOSED.get());
        }
        boolean created = false;
        if (conn == null) {
            conn = this.createConnection();
            this.connections.put(t, conn);
            created = true;
        }
        try {
            this.healthCheck.ensureConnectionValidForCheckout(conn);
            if (created) {
                this.poolStatistics.incrementNumSuccessfulCheckoutsNewConnection();
            } else {
                this.poolStatistics.incrementNumSuccessfulCheckoutsWithoutWaiting();
            }
            return conn;
        }
        catch (LDAPException le) {
            Debug.debugException(le);
            conn.close(null);
            this.connections.remove(t);
            if (created) {
                this.poolStatistics.incrementNumFailedCheckouts();
                throw le;
            }
            try {
                conn = this.createConnection();
                this.healthCheck.ensureConnectionValidForCheckout(conn);
                this.connections.put(t, conn);
                this.poolStatistics.incrementNumSuccessfulCheckoutsNewConnection();
                return conn;
            }
            catch (LDAPException le2) {
                Debug.debugException(le2);
                this.poolStatistics.incrementNumFailedCheckouts();
                if (conn != null) {
                    conn.terminate(null);
                }
                throw le2;
            }
        }
    }

    public void releaseConnection(LDAPConnection connection) {
        if (connection == null) {
            return;
        }
        connection.setConnectionPoolName(this.connectionPoolName);
        if (this.maxConnectionAge > 0L && System.currentTimeMillis() - connection.getConnectTime() > this.maxConnectionAge) {
            connection.setDisconnectInfo(DisconnectType.POOLED_CONNECTION_EXPIRED, null, null);
            this.poolStatistics.incrementNumConnectionsClosedExpired();
            this.handleDefunctConnection(connection);
            return;
        }
        try {
            this.healthCheck.ensureConnectionValidForRelease(connection);
        }
        catch (LDAPException le) {
            this.releaseDefunctConnection(connection);
            return;
        }
        this.poolStatistics.incrementNumReleasedValid();
        if (this.closed) {
            this.close();
        }
    }

    public void releaseDefunctConnection(LDAPConnection connection) {
        if (connection == null) {
            return;
        }
        connection.setConnectionPoolName(this.connectionPoolName);
        this.poolStatistics.incrementNumConnectionsClosedDefunct();
        this.handleDefunctConnection(connection);
    }

    private void handleDefunctConnection(LDAPConnection connection) {
        Thread t = Thread.currentThread();
        connection.setDisconnectInfo(DisconnectType.POOLED_CONNECTION_DEFUNCT, null, null);
        connection.terminate(null);
        this.connections.remove(t);
        if (this.closed) {
            return;
        }
        try {
            LDAPConnection conn = this.createConnection();
            this.connections.put(t, conn);
        }
        catch (LDAPException le) {
            Debug.debugException(le);
        }
    }

    public String getConnectionPoolName() {
        return this.connectionPoolName;
    }

    public void setConnectionPoolName(String connectionPoolName) {
        this.connectionPoolName = connectionPoolName;
    }

    public long getMaxConnectionAgeMillis() {
        return this.maxConnectionAge;
    }

    public void setMaxConnectionAgeMillis(long maxConnectionAge) {
        this.maxConnectionAge = maxConnectionAge > 0L ? maxConnectionAge : 0L;
    }

    public LDAPConnectionPoolHealthCheck getHealthCheck() {
        return this.healthCheck;
    }

    public void setHealthCheck(LDAPConnectionPoolHealthCheck healthCheck) {
        Validator.ensureNotNull(healthCheck);
        this.healthCheck = healthCheck;
    }

    public long getHealthCheckIntervalMillis() {
        return this.healthCheckInterval;
    }

    public void setHealthCheckIntervalMillis(long healthCheckInterval) {
        Validator.ensureTrue(healthCheckInterval > 0L, "LDAPConnectionPool.healthCheckInterval must be greater than 0.");
        this.healthCheckInterval = healthCheckInterval;
        this.healthCheckThread.wakeUp();
    }

    protected void doHealthCheck() {
        Iterator<Map.Entry<Thread, LDAPConnection>> iterator = this.connections.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Thread, LDAPConnection> e = iterator.next();
            Thread t = e.getKey();
            LDAPConnection c = e.getValue();
            if (t.isAlive()) continue;
            c.setDisconnectInfo(DisconnectType.POOLED_CONNECTION_UNNEEDED, null, null);
            c.terminate(null);
            iterator.remove();
        }
    }

    public int getCurrentAvailableConnections() {
        return -1;
    }

    public int getMaximumAvailableConnections() {
        return -1;
    }

    public LDAPConnectionPoolStatistics getConnectionPoolStatistics() {
        return this.poolStatistics;
    }

    protected void finalize() throws Throwable {
        super.finalize();
        this.close();
    }

    public void toString(StringBuilder buffer) {
        buffer.append("LDAPThreadLocalConnectionPool(");
        String name = this.connectionPoolName;
        if (name != null) {
            buffer.append("name='");
            buffer.append(name);
            buffer.append("', ");
        }
        buffer.append("serverSet=");
        this.serverSet.toString(buffer);
        buffer.append(')');
    }
}

