/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.util.ssl.cert;

import com.unboundid.asn1.ASN1BigInteger;
import com.unboundid.asn1.ASN1BitString;
import com.unboundid.asn1.ASN1Element;
import com.unboundid.asn1.ASN1Exception;
import com.unboundid.asn1.ASN1GeneralizedTime;
import com.unboundid.asn1.ASN1Integer;
import com.unboundid.asn1.ASN1ObjectIdentifier;
import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.asn1.ASN1Sequence;
import com.unboundid.asn1.ASN1Set;
import com.unboundid.asn1.ASN1UTCTime;
import com.unboundid.asn1.ASN1UTF8String;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.RDN;
import com.unboundid.ldap.sdk.schema.AttributeTypeDefinition;
import com.unboundid.ldap.sdk.schema.Schema;
import com.unboundid.util.Base64;
import com.unboundid.util.Debug;
import com.unboundid.util.NotMutable;
import com.unboundid.util.NotNull;
import com.unboundid.util.Nullable;
import com.unboundid.util.OID;
import com.unboundid.util.ObjectPair;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import com.unboundid.util.ssl.cert.AuthorityKeyIdentifierExtension;
import com.unboundid.util.ssl.cert.CertException;
import com.unboundid.util.ssl.cert.CertMessages;
import com.unboundid.util.ssl.cert.DecodedPublicKey;
import com.unboundid.util.ssl.cert.EllipticCurvePublicKey;
import com.unboundid.util.ssl.cert.GeneralNamesBuilder;
import com.unboundid.util.ssl.cert.NamedCurve;
import com.unboundid.util.ssl.cert.PublicKeyAlgorithmIdentifier;
import com.unboundid.util.ssl.cert.RSAPublicKey;
import com.unboundid.util.ssl.cert.SignatureAlgorithmIdentifier;
import com.unboundid.util.ssl.cert.SubjectKeyIdentifierExtension;
import com.unboundid.util.ssl.cert.X509CertificateExtension;
import com.unboundid.util.ssl.cert.X509CertificateVersion;
import java.io.ByteArrayInputStream;
import java.io.Serializable;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;

@NotMutable
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public final class X509Certificate
implements Serializable {
    private static final byte TYPE_EXPLICIT_VERSION = -96;
    private static final byte TYPE_IMPLICIT_ISSUER_UNIQUE_ID = -127;
    private static final byte TYPE_IMPLICIT_SUBJECT_UNIQUE_ID = -126;
    private static final byte TYPE_EXPLICIT_EXTENSIONS = -93;
    private static final long serialVersionUID = -4680448103099282243L;
    @Nullable
    private final ASN1BitString issuerUniqueID;
    @NotNull
    private final ASN1BitString signatureValue;
    @NotNull
    private final ASN1BitString encodedPublicKey;
    @Nullable
    private final ASN1BitString subjectUniqueID;
    @Nullable
    private final ASN1Element publicKeyAlgorithmParameters;
    @Nullable
    private final ASN1Element signatureAlgorithmParameters;
    @NotNull
    private final BigInteger serialNumber;
    @NotNull
    private final byte[] x509CertificateBytes;
    @Nullable
    private final DecodedPublicKey decodedPublicKey;
    @NotNull
    private final DN issuerDN;
    @NotNull
    private final DN subjectDN;
    @NotNull
    private final List<X509CertificateExtension> extensions;
    private final long notAfter;
    private final long notBefore;
    @NotNull
    private final OID publicKeyAlgorithmOID;
    @NotNull
    private final OID signatureAlgorithmOID;
    @Nullable
    private final String publicKeyAlgorithmName;
    @Nullable
    private final String signatureAlgorithmName;
    @NotNull
    private final X509CertificateVersion version;

    X509Certificate(@NotNull X509CertificateVersion version, @NotNull BigInteger serialNumber, @NotNull OID signatureAlgorithmOID, @Nullable ASN1Element signatureAlgorithmParameters, @NotNull ASN1BitString signatureValue, @NotNull DN issuerDN, long notBefore, long notAfter, @NotNull DN subjectDN, @NotNull OID publicKeyAlgorithmOID, @Nullable ASN1Element publicKeyAlgorithmParameters, @NotNull ASN1BitString encodedPublicKey, @Nullable DecodedPublicKey decodedPublicKey, @Nullable ASN1BitString issuerUniqueID, @Nullable ASN1BitString subjectUniqueID, X509CertificateExtension ... extensions) throws CertException {
        this.version = version;
        this.serialNumber = serialNumber;
        this.signatureAlgorithmOID = signatureAlgorithmOID;
        this.signatureAlgorithmParameters = signatureAlgorithmParameters;
        this.signatureValue = signatureValue;
        this.issuerDN = issuerDN;
        this.notBefore = notBefore;
        this.notAfter = notAfter;
        this.subjectDN = subjectDN;
        this.publicKeyAlgorithmOID = publicKeyAlgorithmOID;
        this.publicKeyAlgorithmParameters = publicKeyAlgorithmParameters;
        this.encodedPublicKey = encodedPublicKey;
        this.decodedPublicKey = decodedPublicKey;
        this.issuerUniqueID = issuerUniqueID;
        this.subjectUniqueID = subjectUniqueID;
        this.extensions = StaticUtils.toList(extensions);
        SignatureAlgorithmIdentifier signatureAlgorithmIdentifier = SignatureAlgorithmIdentifier.forOID(signatureAlgorithmOID);
        this.signatureAlgorithmName = signatureAlgorithmIdentifier == null ? null : signatureAlgorithmIdentifier.getUserFriendlyName();
        PublicKeyAlgorithmIdentifier publicKeyAlgorithmIdentifier = PublicKeyAlgorithmIdentifier.forOID(publicKeyAlgorithmOID);
        this.publicKeyAlgorithmName = publicKeyAlgorithmIdentifier == null ? null : publicKeyAlgorithmIdentifier.getName();
        this.x509CertificateBytes = this.encode().encode();
    }

    public X509Certificate(@NotNull byte[] encodedCertificate) throws CertException {
        int tbsCertificateElementIndex;
        ASN1Element[] tbsCertificateElements;
        ASN1Element[] certificateElements;
        this.x509CertificateBytes = encodedCertificate;
        try {
            certificateElements = ASN1Sequence.decodeAsSequence(encodedCertificate).elements();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_NOT_SEQUENCE.get(StaticUtils.getExceptionMessage(e)), e);
        }
        if (certificateElements.length != 3) {
            throw new CertException(CertMessages.ERR_CERT_DECODE_UNEXPECTED_SEQUENCE_ELEMENT_COUNT.get(certificateElements.length));
        }
        try {
            tbsCertificateElements = ASN1Sequence.decodeAsSequence(certificateElements[0]).elements();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_FIRST_ELEMENT_NOT_SEQUENCE.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            if ((tbsCertificateElements[0].getType() & 0xFF) == 160) {
                int versionIntValue = ASN1Integer.decodeAsInteger(tbsCertificateElements[0].getValue()).intValue();
                this.version = X509CertificateVersion.valueOf(versionIntValue);
                if (this.version == null) {
                    throw new CertException(CertMessages.ERR_CERT_DECODE_UNSUPPORTED_VERSION.get(new Object[]{this.version}));
                }
                tbsCertificateElementIndex = 1;
            } else {
                this.version = X509CertificateVersion.V1;
                tbsCertificateElementIndex = 0;
            }
        }
        catch (CertException e) {
            Debug.debugException(e);
            throw e;
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_VERSION.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            this.serialNumber = tbsCertificateElements[tbsCertificateElementIndex++].decodeAsBigInteger().getBigIntegerValue();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_SERIAL_NUMBER.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            ASN1Element[] signatureAlgorithmElements = tbsCertificateElements[tbsCertificateElementIndex++].decodeAsSequence().elements();
            this.signatureAlgorithmOID = signatureAlgorithmElements[0].decodeAsObjectIdentifier().getOID();
            this.signatureAlgorithmParameters = signatureAlgorithmElements.length > 1 ? signatureAlgorithmElements[1] : null;
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_SIG_ALG.get(StaticUtils.getExceptionMessage(e)), e);
        }
        SignatureAlgorithmIdentifier signatureAlgorithmIdentifier = SignatureAlgorithmIdentifier.forOID(this.signatureAlgorithmOID);
        this.signatureAlgorithmName = signatureAlgorithmIdentifier == null ? null : signatureAlgorithmIdentifier.getUserFriendlyName();
        try {
            this.issuerDN = X509Certificate.decodeName(tbsCertificateElements[tbsCertificateElementIndex++]);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_ISSUER_DN.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            ASN1Element[] validityElements = tbsCertificateElements[tbsCertificateElementIndex++].decodeAsSequence().elements();
            switch (validityElements[0].getType()) {
                case 23: {
                    this.notBefore = X509Certificate.decodeUTCTime(validityElements[0]);
                    break;
                }
                case 24: {
                    this.notBefore = validityElements[0].decodeAsGeneralizedTime().getTime();
                    break;
                }
                default: {
                    throw new CertException(CertMessages.ERR_CERT_DECODE_NOT_BEFORE_UNEXPECTED_TYPE.get(StaticUtils.toHex(validityElements[0].getType()), StaticUtils.toHex((byte)23), StaticUtils.toHex((byte)24)));
                }
            }
            switch (validityElements[1].getType()) {
                case 23: {
                    this.notAfter = X509Certificate.decodeUTCTime(validityElements[1]);
                    break;
                }
                case 24: {
                    this.notAfter = validityElements[1].decodeAsGeneralizedTime().getTime();
                    break;
                }
                default: {
                    throw new CertException(CertMessages.ERR_CERT_DECODE_NOT_AFTER_UNEXPECTED_TYPE.get(StaticUtils.toHex(validityElements[0].getType()), StaticUtils.toHex((byte)23), StaticUtils.toHex((byte)24)));
                }
            }
        }
        catch (CertException e) {
            Debug.debugException(e);
            throw e;
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_COULD_NOT_PARSE_VALIDITY.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            this.subjectDN = X509Certificate.decodeName(tbsCertificateElements[tbsCertificateElementIndex++]);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_SUBJECT_DN.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            ASN1Element[] subjectPublicKeyInfoElements = tbsCertificateElements[tbsCertificateElementIndex++].decodeAsSequence().elements();
            ASN1Element[] publicKeyAlgorithmElements = subjectPublicKeyInfoElements[0].decodeAsSequence().elements();
            this.publicKeyAlgorithmOID = publicKeyAlgorithmElements[0].decodeAsObjectIdentifier().getOID();
            this.publicKeyAlgorithmParameters = publicKeyAlgorithmElements.length > 1 ? publicKeyAlgorithmElements[1] : null;
            this.encodedPublicKey = subjectPublicKeyInfoElements[1].decodeAsBitString();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_PUBLIC_KEY_INFO.get(StaticUtils.getExceptionMessage(e)), e);
        }
        PublicKeyAlgorithmIdentifier publicKeyAlgorithmIdentifier = PublicKeyAlgorithmIdentifier.forOID(this.publicKeyAlgorithmOID);
        if (publicKeyAlgorithmIdentifier == null) {
            this.publicKeyAlgorithmName = null;
            this.decodedPublicKey = null;
        } else {
            this.publicKeyAlgorithmName = publicKeyAlgorithmIdentifier.getName();
            DecodedPublicKey pk = null;
            switch (publicKeyAlgorithmIdentifier) {
                case RSA: {
                    try {
                        pk = new RSAPublicKey(this.encodedPublicKey);
                    }
                    catch (Exception e) {
                        Debug.debugException(e);
                    }
                    break;
                }
                case EC: {
                    try {
                        pk = new EllipticCurvePublicKey(this.encodedPublicKey);
                        break;
                    }
                    catch (Exception e) {
                        Debug.debugException(e);
                    }
                }
            }
            this.decodedPublicKey = pk;
        }
        ASN1BitString issuerID = null;
        ASN1BitString subjectID = null;
        ArrayList<X509CertificateExtension> extList = new ArrayList<X509CertificateExtension>(10);
        while (tbsCertificateElementIndex < tbsCertificateElements.length) {
            switch (tbsCertificateElements[tbsCertificateElementIndex].getType()) {
                case -127: {
                    try {
                        issuerID = tbsCertificateElements[tbsCertificateElementIndex].decodeAsBitString();
                        break;
                    }
                    catch (Exception e) {
                        Debug.debugException(e);
                        throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_ISSUER_UNIQUE_ID.get(StaticUtils.getExceptionMessage(e)), e);
                    }
                }
                case -126: {
                    try {
                        subjectID = tbsCertificateElements[tbsCertificateElementIndex].decodeAsBitString();
                        break;
                    }
                    catch (Exception e) {
                        Debug.debugException(e);
                        throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_SUBJECT_UNIQUE_ID.get(StaticUtils.getExceptionMessage(e)), e);
                    }
                }
                case -93: {
                    try {
                        ASN1Element[] extensionElements;
                        for (ASN1Element extensionElement : extensionElements = ASN1Sequence.decodeAsSequence(tbsCertificateElements[tbsCertificateElementIndex].getValue()).elements()) {
                            extList.add(X509CertificateExtension.decode(extensionElement));
                        }
                        break;
                    }
                    catch (Exception e) {
                        Debug.debugException(e);
                        throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_EXTENSION.get(StaticUtils.getExceptionMessage(e)), e);
                    }
                }
            }
            ++tbsCertificateElementIndex;
        }
        this.issuerUniqueID = issuerID;
        this.subjectUniqueID = subjectID;
        this.extensions = Collections.unmodifiableList(extList);
        try {
            ASN1Element[] signatureAlgorithmElements = certificateElements[1].decodeAsSequence().elements();
            OID oid = signatureAlgorithmElements[0].decodeAsObjectIdentifier().getOID();
            if (!oid.equals(this.signatureAlgorithmOID)) {
                throw new CertException(CertMessages.ERR_CERT_DECODE_SIG_ALG_MISMATCH.get(this.signatureAlgorithmOID.toString(), oid.toString()));
            }
        }
        catch (CertException e) {
            Debug.debugException(e);
            throw e;
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_SIG_ALG.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            this.signatureValue = certificateElements[2].decodeAsBitString();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_SIG_VALUE.get(StaticUtils.getExceptionMessage(e)), e);
        }
    }

    @NotNull
    static DN decodeName(@NotNull ASN1Element element) throws CertException {
        ASN1Element[] rdnElements;
        Schema schema;
        try {
            schema = Schema.getDefaultStandardSchema();
        }
        catch (Exception e) {
            Debug.debugException(e);
            schema = null;
        }
        try {
            rdnElements = ASN1Sequence.decodeAsSequence(element).elements();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_DECODE_NAME_NOT_SEQUENCE.get(StaticUtils.getExceptionMessage(e)), e);
        }
        ArrayList<RDN> rdns = new ArrayList<RDN>(rdnElements.length);
        for (int i = 0; i < rdnElements.length; ++i) {
            try {
                ASN1Element[] attributeSetElements = rdnElements[i].decodeAsSet().elements();
                String[] attributeNames = new String[attributeSetElements.length];
                byte[][] attributeValues = new byte[attributeSetElements.length][];
                for (int j = 0; j < attributeSetElements.length; ++j) {
                    ASN1Element[] attributeTypeAndValueElements = ASN1Sequence.decodeAsSequence(attributeSetElements[j]).elements();
                    OID attributeTypeOID = attributeTypeAndValueElements[0].decodeAsObjectIdentifier().getOID();
                    AttributeTypeDefinition attributeType = schema.getAttributeType(attributeTypeOID.toString());
                    attributeNames[j] = attributeType == null ? attributeTypeOID.toString() : attributeType.getNameOrOID().toUpperCase();
                    attributeValues[j] = attributeTypeAndValueElements[1].decodeAsOctetString().getValue();
                }
                rdns.add(new RDN(attributeNames, attributeValues, schema));
                continue;
            }
            catch (Exception e) {
                Debug.debugException(e);
                throw new CertException(CertMessages.ERR_CERT_DECODE_CANNOT_PARSE_NAME_SEQUENCE_ELEMENT.get(i, StaticUtils.getExceptionMessage(e)), e);
            }
        }
        Collections.reverse(rdns);
        return new DN(rdns);
    }

    private static long decodeUTCTime(@NotNull ASN1Element element) throws ASN1Exception {
        long timeValue = ASN1UTCTime.decodeAsUTCTime(element).getTime();
        GregorianCalendar calendar = new GregorianCalendar();
        calendar.setTimeInMillis(timeValue);
        int year = calendar.get(1);
        if (year < 1949) {
            calendar.set(1, year + 100);
        } else if (year > 2050) {
            calendar.set(1, year - 100);
        }
        return calendar.getTimeInMillis();
    }

    @NotNull
    ASN1Element encode() throws CertException {
        try {
            ArrayList<ASN1Element> tbsCertificateElements = new ArrayList<ASN1Element>(10);
            if (this.version != X509CertificateVersion.V1) {
                tbsCertificateElements.add(new ASN1Element(-96, new ASN1Integer(this.version.getIntValue()).encode()));
            }
            tbsCertificateElements.add(new ASN1BigInteger(this.serialNumber));
            if (this.signatureAlgorithmParameters == null) {
                tbsCertificateElements.add(new ASN1Sequence(new ASN1ObjectIdentifier(this.signatureAlgorithmOID)));
            } else {
                tbsCertificateElements.add(new ASN1Sequence(new ASN1ObjectIdentifier(this.signatureAlgorithmOID), this.signatureAlgorithmParameters));
            }
            tbsCertificateElements.add(X509Certificate.encodeName(this.issuerDN));
            tbsCertificateElements.add(X509Certificate.encodeValiditySequence(this.notBefore, this.notAfter));
            tbsCertificateElements.add(X509Certificate.encodeName(this.subjectDN));
            if (this.publicKeyAlgorithmParameters == null) {
                tbsCertificateElements.add(new ASN1Sequence(new ASN1Sequence(new ASN1ObjectIdentifier(this.publicKeyAlgorithmOID)), this.encodedPublicKey));
            } else {
                tbsCertificateElements.add(new ASN1Sequence(new ASN1Sequence(new ASN1ObjectIdentifier(this.publicKeyAlgorithmOID), this.publicKeyAlgorithmParameters), this.encodedPublicKey));
            }
            if (this.issuerUniqueID != null) {
                tbsCertificateElements.add(new ASN1BitString(-127, this.issuerUniqueID.getBits()));
            }
            if (this.subjectUniqueID != null) {
                tbsCertificateElements.add(new ASN1BitString(-126, this.subjectUniqueID.getBits()));
            }
            if (!this.extensions.isEmpty()) {
                ArrayList<ASN1Sequence> extensionElements = new ArrayList<ASN1Sequence>(this.extensions.size());
                for (X509CertificateExtension e : this.extensions) {
                    extensionElements.add(e.encode());
                }
                tbsCertificateElements.add(new ASN1Element(-93, new ASN1Sequence(extensionElements).encode()));
            }
            ArrayList<ASN1Element> certificateElements = new ArrayList<ASN1Element>(3);
            certificateElements.add(new ASN1Sequence(tbsCertificateElements));
            if (this.signatureAlgorithmParameters == null) {
                certificateElements.add(new ASN1Sequence(new ASN1ObjectIdentifier(this.signatureAlgorithmOID)));
            } else {
                certificateElements.add(new ASN1Sequence(new ASN1ObjectIdentifier(this.signatureAlgorithmOID), this.signatureAlgorithmParameters));
            }
            certificateElements.add(this.signatureValue);
            return new ASN1Sequence(certificateElements);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_ENCODE_ERROR.get(this.toString(), StaticUtils.getExceptionMessage(e)), e);
        }
    }

    @NotNull
    static ASN1Element encodeName(@NotNull DN dn) throws CertException {
        Schema schema;
        try {
            schema = Schema.getDefaultStandardSchema();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_ENCODE_NAME_CANNOT_GET_SCHEMA.get(String.valueOf(dn), StaticUtils.getExceptionMessage(e)), e);
        }
        RDN[] rdns = dn.getRDNs();
        ArrayList<ASN1Set> rdnSequenceElements = new ArrayList<ASN1Set>(rdns.length);
        for (int i = rdns.length - 1; i >= 0; --i) {
            RDN rdn = rdns[i];
            String[] names = rdn.getAttributeNames();
            String[] values = rdn.getAttributeValues();
            ArrayList<ASN1Sequence> rdnElements = new ArrayList<ASN1Sequence>(names.length);
            for (int j = 0; j < names.length; ++j) {
                AttributeTypeDefinition at = schema.getAttributeType(names[j]);
                if (at == null) {
                    throw new CertException(CertMessages.ERR_CERT_ENCODE_NAME_UNKNOWN_ATTR_TYPE.get(String.valueOf(dn), names[j]));
                }
                try {
                    rdnElements.add(new ASN1Sequence(new ASN1ObjectIdentifier(at.getOID()), new ASN1UTF8String(values[j])));
                    continue;
                }
                catch (Exception e) {
                    Debug.debugException(e);
                    throw new CertException(CertMessages.ERR_CERT_ENCODE_NAME_ERROR.get(String.valueOf(dn), StaticUtils.getExceptionMessage(e)), e);
                }
            }
            rdnSequenceElements.add(new ASN1Set(rdnElements));
        }
        return new ASN1Sequence(rdnSequenceElements);
    }

    @NotNull
    static ASN1Sequence encodeValiditySequence(long notBefore, long notAfter) {
        GregorianCalendar notBeforeCalendar = new GregorianCalendar();
        notBeforeCalendar.setTimeInMillis(notBefore);
        int notBeforeYear = notBeforeCalendar.get(1);
        GregorianCalendar notAfterCalendar = new GregorianCalendar();
        notAfterCalendar.setTimeInMillis(notAfter);
        int notAfterYear = notAfterCalendar.get(1);
        if (notBeforeYear >= 1950 && notBeforeYear <= 2049 && notAfterYear >= 1950 && notAfterYear <= 2049) {
            return new ASN1Sequence(new ASN1UTCTime(notBefore), new ASN1UTCTime(notAfter));
        }
        return new ASN1Sequence(new ASN1GeneralizedTime(notBefore), new ASN1GeneralizedTime(notAfter));
    }

    @NotNull
    public static ObjectPair<X509Certificate, KeyPair> generateSelfSignedCertificate(@NotNull SignatureAlgorithmIdentifier signatureAlgorithm, @NotNull PublicKeyAlgorithmIdentifier publicKeyAlgorithm, int keySizeBits, @NotNull DN subjectDN, long notBefore, long notAfter, X509CertificateExtension ... extensions) throws CertException {
        KeyPair keyPair;
        KeyPairGenerator keyPairGenerator;
        try {
            keyPairGenerator = KeyPairGenerator.getInstance(publicKeyAlgorithm.getName());
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SELF_SIGNED_CANNOT_GET_KEY_GENERATOR.get(publicKeyAlgorithm.getName(), StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            keyPairGenerator.initialize(keySizeBits);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SELF_SIGNED_INVALID_KEY_SIZE.get(keySizeBits, publicKeyAlgorithm.getName(), StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            keyPair = keyPairGenerator.generateKeyPair();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SELF_SIGNED_CANNOT_GENERATE_KEY_PAIR.get(keySizeBits, publicKeyAlgorithm.getName(), StaticUtils.getExceptionMessage(e)), e);
        }
        X509Certificate certificate = X509Certificate.generateSelfSignedCertificate(signatureAlgorithm, keyPair, subjectDN, notBefore, notAfter, extensions);
        return new ObjectPair<X509Certificate, KeyPair>(certificate, keyPair);
    }

    @NotNull
    public static X509Certificate generateSelfSignedCertificate(@NotNull SignatureAlgorithmIdentifier signatureAlgorithm, @NotNull KeyPair keyPair, @NotNull DN subjectDN, long notBefore, long notAfter, X509CertificateExtension ... extensions) throws CertException {
        byte[] subjectKeyIdentifier;
        ASN1BitString encodedPublicKey;
        ASN1Element publicKeyAlgorithmParameters;
        OID publicKeyAlgorithmOID;
        DecodedPublicKey decodedPublicKey = null;
        try {
            ASN1Element[] pkElements = ASN1Sequence.decodeAsSequence(keyPair.getPublic().getEncoded()).elements();
            ASN1Element[] pkAlgIDElements = ASN1Sequence.decodeAsSequence(pkElements[0]).elements();
            publicKeyAlgorithmOID = pkAlgIDElements[0].decodeAsObjectIdentifier().getOID();
            publicKeyAlgorithmParameters = pkAlgIDElements.length == 1 ? null : pkAlgIDElements[1];
            encodedPublicKey = pkElements[1].decodeAsBitString();
            try {
                if (publicKeyAlgorithmOID.equals(PublicKeyAlgorithmIdentifier.RSA.getOID())) {
                    decodedPublicKey = new RSAPublicKey(encodedPublicKey);
                } else if (publicKeyAlgorithmOID.equals(PublicKeyAlgorithmIdentifier.EC.getOID())) {
                    decodedPublicKey = new EllipticCurvePublicKey(encodedPublicKey);
                }
            }
            catch (Exception e) {
                Debug.debugException(e);
            }
            MessageDigest sha256 = MessageDigest.getInstance("SHA-1");
            subjectKeyIdentifier = sha256.digest(encodedPublicKey.getBytes());
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SELF_SIGNED_CANNOT_PARSE_KEY_PAIR.get(StaticUtils.getExceptionMessage(e)), e);
        }
        ArrayList<X509CertificateExtension> extensionList = new ArrayList<X509CertificateExtension>(10);
        extensionList.add(new SubjectKeyIdentifierExtension(false, new ASN1OctetString(subjectKeyIdentifier)));
        if (extensions != null) {
            for (X509CertificateExtension e : extensions) {
                if (e.getOID().equals(SubjectKeyIdentifierExtension.SUBJECT_KEY_IDENTIFIER_OID)) continue;
                extensionList.add(e);
            }
        }
        X509CertificateExtension[] allExtensions = new X509CertificateExtension[extensionList.size()];
        extensionList.toArray(allExtensions);
        BigInteger serialNumber = X509Certificate.generateSerialNumber();
        ASN1BitString encodedSignature = X509Certificate.generateSignature(signatureAlgorithm, keyPair.getPrivate(), serialNumber, subjectDN, notBefore, notAfter, subjectDN, publicKeyAlgorithmOID, publicKeyAlgorithmParameters, encodedPublicKey, allExtensions);
        return new X509Certificate(X509CertificateVersion.V3, serialNumber, signatureAlgorithm.getOID(), null, encodedSignature, subjectDN, notBefore, notAfter, subjectDN, publicKeyAlgorithmOID, publicKeyAlgorithmParameters, encodedPublicKey, decodedPublicKey, null, null, allExtensions);
    }

    @NotNull
    public static X509Certificate generateIssuerSignedCertificate(@NotNull SignatureAlgorithmIdentifier signatureAlgorithm, @NotNull X509Certificate issuerCertificate, @NotNull PrivateKey issuerPrivateKey, @NotNull OID publicKeyAlgorithmOID, @Nullable ASN1Element publicKeyAlgorithmParameters, @NotNull ASN1BitString encodedPublicKey, @Nullable DecodedPublicKey decodedPublicKey, @NotNull DN subjectDN, long notBefore, long notAfter, X509CertificateExtension ... extensions) throws CertException {
        byte[] subjectKeyIdentifier;
        try {
            MessageDigest sha256 = MessageDigest.getInstance("SHA-1");
            subjectKeyIdentifier = sha256.digest(encodedPublicKey.getBytes());
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_ISSUER_SIGNED_CANNOT_GENERATE_KEY_ID.get(StaticUtils.getExceptionMessage(e)), e);
        }
        ASN1OctetString authorityKeyIdentifier = null;
        for (X509CertificateExtension e : issuerCertificate.extensions) {
            if (!(e instanceof SubjectKeyIdentifierExtension)) continue;
            authorityKeyIdentifier = ((SubjectKeyIdentifierExtension)e).getKeyIdentifier();
        }
        ArrayList<X509CertificateExtension> extensionList = new ArrayList<X509CertificateExtension>(10);
        extensionList.add(new SubjectKeyIdentifierExtension(false, new ASN1OctetString(subjectKeyIdentifier)));
        if (authorityKeyIdentifier == null) {
            extensionList.add(new AuthorityKeyIdentifierExtension(false, null, new GeneralNamesBuilder().addDirectoryName(issuerCertificate.subjectDN).build(), issuerCertificate.serialNumber));
        } else {
            extensionList.add(new AuthorityKeyIdentifierExtension(false, authorityKeyIdentifier, null, null));
        }
        if (extensions != null) {
            for (X509CertificateExtension e : extensions) {
                if (e.getOID().equals(SubjectKeyIdentifierExtension.SUBJECT_KEY_IDENTIFIER_OID) || e.getOID().equals(AuthorityKeyIdentifierExtension.AUTHORITY_KEY_IDENTIFIER_OID)) continue;
                extensionList.add(e);
            }
        }
        X509CertificateExtension[] allExtensions = new X509CertificateExtension[extensionList.size()];
        extensionList.toArray(allExtensions);
        BigInteger serialNumber = X509Certificate.generateSerialNumber();
        ASN1BitString encodedSignature = X509Certificate.generateSignature(signatureAlgorithm, issuerPrivateKey, serialNumber, issuerCertificate.subjectDN, notBefore, notAfter, subjectDN, publicKeyAlgorithmOID, publicKeyAlgorithmParameters, encodedPublicKey, allExtensions);
        return new X509Certificate(X509CertificateVersion.V3, serialNumber, signatureAlgorithm.getOID(), null, encodedSignature, issuerCertificate.subjectDN, notBefore, notAfter, subjectDN, publicKeyAlgorithmOID, publicKeyAlgorithmParameters, encodedPublicKey, decodedPublicKey, null, null, allExtensions);
    }

    @NotNull
    private static BigInteger generateSerialNumber() {
        UUID uuid = UUID.randomUUID();
        long msb = uuid.getMostSignificantBits() & Long.MAX_VALUE;
        long lsb = uuid.getLeastSignificantBits() & Long.MAX_VALUE;
        return BigInteger.valueOf(msb).shiftLeft(64).add(BigInteger.valueOf(lsb));
    }

    @NotNull
    private static ASN1BitString generateSignature(@NotNull SignatureAlgorithmIdentifier signatureAlgorithm, @NotNull PrivateKey privateKey, @NotNull BigInteger serialNumber, @NotNull DN issuerDN, long notBefore, long notAfter, @NotNull DN subjectDN, @NotNull OID publicKeyAlgorithmOID, @Nullable ASN1Element publicKeyAlgorithmParameters, @NotNull ASN1BitString encodedPublicKey, X509CertificateExtension ... extensions) throws CertException {
        Signature signature;
        try {
            signature = Signature.getInstance(signatureAlgorithm.getJavaName());
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SIGNATURE_CANNOT_GET_SIGNATURE_GENERATOR.get(signatureAlgorithm.getJavaName(), StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            signature.initSign(privateKey);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SIGNATURE_CANNOT_INIT_SIGNATURE_GENERATOR.get(signatureAlgorithm.getJavaName(), StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            ArrayList<ASN1Element> tbsCertificateElements = new ArrayList<ASN1Element>(8);
            tbsCertificateElements.add(new ASN1Element(-96, new ASN1Integer(X509CertificateVersion.V3.getIntValue()).encode()));
            tbsCertificateElements.add(new ASN1BigInteger(serialNumber));
            tbsCertificateElements.add(new ASN1Sequence(new ASN1ObjectIdentifier(signatureAlgorithm.getOID())));
            tbsCertificateElements.add(X509Certificate.encodeName(issuerDN));
            tbsCertificateElements.add(X509Certificate.encodeValiditySequence(notBefore, notAfter));
            tbsCertificateElements.add(X509Certificate.encodeName(subjectDN));
            if (publicKeyAlgorithmParameters == null) {
                tbsCertificateElements.add(new ASN1Sequence(new ASN1Sequence(new ASN1ObjectIdentifier(publicKeyAlgorithmOID)), encodedPublicKey));
            } else {
                tbsCertificateElements.add(new ASN1Sequence(new ASN1Sequence(new ASN1ObjectIdentifier(publicKeyAlgorithmOID), publicKeyAlgorithmParameters), encodedPublicKey));
            }
            ArrayList<ASN1Sequence> extensionElements = new ArrayList<ASN1Sequence>(extensions.length);
            for (X509CertificateExtension e : extensions) {
                extensionElements.add(e.encode());
            }
            tbsCertificateElements.add(new ASN1Element(-93, new ASN1Sequence(extensionElements).encode()));
            byte[] tbsCertificateBytes = new ASN1Sequence(tbsCertificateElements).encode();
            signature.update(tbsCertificateBytes);
            byte[] signatureBytes = signature.sign();
            return new ASN1BitString(ASN1BitString.getBitsForBytes(signatureBytes));
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SIGNATURE_CANNOT_COMPUTE.get(signatureAlgorithm.getJavaName(), StaticUtils.getExceptionMessage(e)), e);
        }
    }

    @NotNull
    public byte[] getX509CertificateBytes() {
        return this.x509CertificateBytes;
    }

    @NotNull
    public X509CertificateVersion getVersion() {
        return this.version;
    }

    @NotNull
    public BigInteger getSerialNumber() {
        return this.serialNumber;
    }

    @NotNull
    public OID getSignatureAlgorithmOID() {
        return this.signatureAlgorithmOID;
    }

    @Nullable
    public String getSignatureAlgorithmName() {
        return this.signatureAlgorithmName;
    }

    @NotNull
    public String getSignatureAlgorithmNameOrOID() {
        if (this.signatureAlgorithmName != null) {
            return this.signatureAlgorithmName;
        }
        return this.signatureAlgorithmOID.toString();
    }

    @Nullable
    public ASN1Element getSignatureAlgorithmParameters() {
        return this.signatureAlgorithmParameters;
    }

    @NotNull
    public DN getIssuerDN() {
        return this.issuerDN;
    }

    public long getNotBeforeTime() {
        return this.notBefore;
    }

    @NotNull
    public Date getNotBeforeDate() {
        return new Date(this.notBefore);
    }

    public long getNotAfterTime() {
        return this.notAfter;
    }

    @NotNull
    public Date getNotAfterDate() {
        return new Date(this.notAfter);
    }

    public boolean isWithinValidityWindow() {
        return this.isWithinValidityWindow(System.currentTimeMillis());
    }

    public boolean isWithinValidityWindow(@NotNull Date date) {
        return this.isWithinValidityWindow(date.getTime());
    }

    public boolean isWithinValidityWindow(long time) {
        return time >= this.notBefore && time <= this.notAfter;
    }

    @NotNull
    public DN getSubjectDN() {
        return this.subjectDN;
    }

    @NotNull
    public OID getPublicKeyAlgorithmOID() {
        return this.publicKeyAlgorithmOID;
    }

    @Nullable
    public String getPublicKeyAlgorithmName() {
        return this.publicKeyAlgorithmName;
    }

    @NotNull
    public String getPublicKeyAlgorithmNameOrOID() {
        if (this.publicKeyAlgorithmName != null) {
            return this.publicKeyAlgorithmName;
        }
        return this.publicKeyAlgorithmOID.toString();
    }

    @Nullable
    public ASN1Element getPublicKeyAlgorithmParameters() {
        return this.publicKeyAlgorithmParameters;
    }

    @NotNull
    public ASN1BitString getEncodedPublicKey() {
        return this.encodedPublicKey;
    }

    @Nullable
    public DecodedPublicKey getDecodedPublicKey() {
        return this.decodedPublicKey;
    }

    @Nullable
    public ASN1BitString getIssuerUniqueID() {
        return this.issuerUniqueID;
    }

    @Nullable
    public ASN1BitString getSubjectUniqueID() {
        return this.subjectUniqueID;
    }

    @NotNull
    public List<X509CertificateExtension> getExtensions() {
        return this.extensions;
    }

    @NotNull
    public ASN1BitString getSignatureValue() {
        return this.signatureValue;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void verifySignature(@Nullable X509Certificate issuerCertificate) throws CertException {
        Signature signature;
        SignatureAlgorithmIdentifier signatureAlgorithm;
        PublicKey publicKey;
        X509Certificate issuer;
        if (issuerCertificate == null) {
            if (!this.isSelfSigned()) throw new CertException(CertMessages.ERR_CERT_VERIFY_SIGNATURE_ISSUER_CERT_NOT_PROVIDED.get());
            issuer = this;
        } else {
            issuer = issuerCertificate;
        }
        try {
            publicKey = issuer.toCertificate().getPublicKey();
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_VERIFY_SIGNATURE_CANNOT_GET_PUBLIC_KEY.get(StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            signatureAlgorithm = SignatureAlgorithmIdentifier.forOID(this.signatureAlgorithmOID);
            signature = Signature.getInstance(signatureAlgorithm.getJavaName());
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_VERIFY_SIGNATURE_CANNOT_GET_SIGNATURE_VERIFIER.get(this.getSignatureAlgorithmNameOrOID(), StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            signature.initVerify(publicKey);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_VERIFY_SIGNATURE_CANNOT_INIT_SIGNATURE_VERIFIER.get(signatureAlgorithm.getJavaName(), StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            ASN1Element[] x509CertificateElements = ASN1Sequence.decodeAsSequence(this.x509CertificateBytes).elements();
            byte[] tbsCertificateBytes = x509CertificateElements[0].encode();
            signature.update(tbsCertificateBytes);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_GEN_SIGNATURE_CANNOT_COMPUTE.get(signatureAlgorithm.getJavaName(), StaticUtils.getExceptionMessage(e)), e);
        }
        try {
            if (signature.verify(this.signatureValue.getBytes())) return;
            throw new CertException(CertMessages.ERR_CERT_VERIFY_SIGNATURE_NOT_VALID.get(this.subjectDN));
        }
        catch (CertException ce) {
            Debug.debugException(ce);
            throw ce;
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_VERIFY_SIGNATURE_ERROR.get(this.subjectDN, StaticUtils.getExceptionMessage(e)), e);
        }
    }

    @NotNull
    public byte[] getSHA1Fingerprint() throws CertException {
        return this.getFingerprint("SHA-1");
    }

    @NotNull
    public byte[] getSHA256Fingerprint() throws CertException {
        return this.getFingerprint("SHA-256");
    }

    @NotNull
    private byte[] getFingerprint(@NotNull String digestAlgorithm) throws CertException {
        try {
            MessageDigest digest = MessageDigest.getInstance(digestAlgorithm);
            return digest.digest(this.x509CertificateBytes);
        }
        catch (Exception e) {
            Debug.debugException(e);
            throw new CertException(CertMessages.ERR_CERT_CANNOT_COMPUTE_FINGERPRINT.get(digestAlgorithm, StaticUtils.getExceptionMessage(e)), e);
        }
    }

    public boolean isSelfSigned() {
        AuthorityKeyIdentifierExtension akie = null;
        SubjectKeyIdentifierExtension skie = null;
        for (X509CertificateExtension e : this.extensions) {
            if (e instanceof AuthorityKeyIdentifierExtension) {
                akie = (AuthorityKeyIdentifierExtension)e;
                continue;
            }
            if (!(e instanceof SubjectKeyIdentifierExtension)) continue;
            skie = (SubjectKeyIdentifierExtension)e;
        }
        if (akie != null && skie != null) {
            return akie.getKeyIdentifier() != null && Arrays.equals(akie.getKeyIdentifier().getValue(), skie.getKeyIdentifier().getValue());
        }
        return this.subjectDN.equals(this.issuerDN);
    }

    public boolean isIssuerFor(@NotNull X509Certificate c) {
        return this.isIssuerFor(c, null);
    }

    public boolean isIssuerFor(@NotNull X509Certificate c, @Nullable StringBuilder nonMatchReason) {
        if (!c.issuerDN.equals(this.subjectDN)) {
            if (nonMatchReason != null) {
                nonMatchReason.append(CertMessages.INFO_CERT_IS_ISSUER_FOR_DN_MISMATCH.get(this.subjectDN, c.subjectDN, this.issuerDN));
            }
            return false;
        }
        byte[] authorityKeyIdentifier = null;
        for (X509CertificateExtension extension : c.extensions) {
            AuthorityKeyIdentifierExtension akie;
            if (!(extension instanceof AuthorityKeyIdentifierExtension) || (akie = (AuthorityKeyIdentifierExtension)extension).getKeyIdentifier() == null) continue;
            authorityKeyIdentifier = akie.getKeyIdentifier().getValue();
            break;
        }
        if (authorityKeyIdentifier != null) {
            boolean matchFound = false;
            for (X509CertificateExtension extension : this.extensions) {
                if (!(extension instanceof SubjectKeyIdentifierExtension)) continue;
                SubjectKeyIdentifierExtension skie = (SubjectKeyIdentifierExtension)extension;
                matchFound = Arrays.equals(authorityKeyIdentifier, skie.getKeyIdentifier().getValue());
                break;
            }
            if (!matchFound) {
                if (nonMatchReason != null) {
                    nonMatchReason.append(CertMessages.INFO_CERT_IS_ISSUER_FOR_KEY_ID_MISMATCH.get(this.subjectDN, c.subjectDN));
                }
                return false;
            }
        }
        return true;
    }

    @NotNull
    public Certificate toCertificate() throws CertificateException {
        return CertificateFactory.getInstance("X.509").generateCertificate(new ByteArrayInputStream(this.x509CertificateBytes));
    }

    @NotNull
    public String toString() {
        StringBuilder buffer = new StringBuilder();
        this.toString(buffer);
        return buffer.toString();
    }

    public void toString(@NotNull StringBuilder buffer) {
        buffer.append("X509Certificate(version='");
        buffer.append(this.version.getName());
        buffer.append("', serialNumber='");
        StaticUtils.toHex(this.serialNumber.toByteArray(), ":", buffer);
        buffer.append("', signatureAlgorithmOID='");
        buffer.append(this.signatureAlgorithmOID.toString());
        buffer.append('\'');
        if (this.signatureAlgorithmName != null) {
            buffer.append(", signatureAlgorithmName='");
            buffer.append(this.signatureAlgorithmName);
            buffer.append('\'');
        }
        buffer.append(", issuerDN='");
        buffer.append(this.issuerDN.toString());
        buffer.append("', notBefore='");
        buffer.append(StaticUtils.encodeGeneralizedTime(this.notBefore));
        buffer.append("', notAfter='");
        buffer.append(StaticUtils.encodeGeneralizedTime(this.notAfter));
        buffer.append("', subjectDN='");
        buffer.append(this.subjectDN.toString());
        buffer.append("', publicKeyAlgorithmOID='");
        buffer.append(this.publicKeyAlgorithmOID.toString());
        buffer.append('\'');
        if (this.publicKeyAlgorithmName != null) {
            buffer.append(", publicKeyAlgorithmName='");
            buffer.append(this.publicKeyAlgorithmName);
            buffer.append('\'');
        }
        buffer.append(", subjectPublicKey=");
        if (this.decodedPublicKey == null) {
            buffer.append('\'');
            try {
                StaticUtils.toHex(this.encodedPublicKey.getBytes(), ":", buffer);
            }
            catch (Exception e) {
                Debug.debugException(e);
                this.encodedPublicKey.toString(buffer);
            }
            buffer.append('\'');
        } else {
            this.decodedPublicKey.toString(buffer);
            if (this.decodedPublicKey instanceof EllipticCurvePublicKey) {
                try {
                    OID namedCurveOID = this.publicKeyAlgorithmParameters.decodeAsObjectIdentifier().getOID();
                    buffer.append(", ellipticCurvePublicKeyParameters=namedCurve='");
                    buffer.append(NamedCurve.getNameOrOID(namedCurveOID));
                    buffer.append('\'');
                }
                catch (Exception e) {
                    Debug.debugException(e);
                }
            }
        }
        if (this.issuerUniqueID != null) {
            buffer.append(", issuerUniqueID='");
            buffer.append(this.issuerUniqueID.toString());
            buffer.append('\'');
        }
        if (this.subjectUniqueID != null) {
            buffer.append(", subjectUniqueID='");
            buffer.append(this.subjectUniqueID.toString());
            buffer.append('\'');
        }
        if (!this.extensions.isEmpty()) {
            buffer.append(", extensions={");
            Iterator<X509CertificateExtension> iterator = this.extensions.iterator();
            while (iterator.hasNext()) {
                iterator.next().toString(buffer);
                if (!iterator.hasNext()) continue;
                buffer.append(", ");
            }
            buffer.append('}');
        }
        buffer.append(", signatureValue='");
        try {
            StaticUtils.toHex(this.signatureValue.getBytes(), ":", buffer);
        }
        catch (Exception e) {
            Debug.debugException(e);
            buffer.append(this.signatureValue.toString());
        }
        buffer.append("')");
    }

    @NotNull
    public List<String> toPEM() {
        ArrayList<String> lines = new ArrayList<String>(10);
        lines.add("-----BEGIN CERTIFICATE-----");
        String certBase64 = Base64.encode(this.x509CertificateBytes);
        lines.addAll(StaticUtils.wrapLine(certBase64, 64));
        lines.add("-----END CERTIFICATE-----");
        return Collections.unmodifiableList(lines);
    }

    @NotNull
    public String toPEMString() {
        StringBuilder buffer = new StringBuilder();
        buffer.append("-----BEGIN CERTIFICATE-----");
        buffer.append(StaticUtils.EOL);
        String certBase64 = Base64.encode(this.x509CertificateBytes);
        for (String line : StaticUtils.wrapLine(certBase64, 64)) {
            buffer.append(line);
            buffer.append(StaticUtils.EOL);
        }
        buffer.append("-----END CERTIFICATE-----");
        buffer.append(StaticUtils.EOL);
        return buffer.toString();
    }
}

