/*
 * Copyright 2009-2024 Ping Identity Corporation
 * All Rights Reserved.
 */
/*
 * Copyright 2009-2024 Ping Identity Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2009-2024 Ping Identity Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (GPLv2 only)
 * or the terms of the GNU Lesser General Public License (LGPLv2.1 only)
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>.
 */
package com.unboundid.ldap.sdk.unboundidds.logs;



import com.unboundid.util.NotNull;
import com.unboundid.util.Nullable;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;



/**
 * This enum contains the set of error log categories defined in the Directory
 * Server.
 * <BR>
 * <BLOCKQUOTE>
 *   <B>NOTE:</B>  This class, and other classes within the
 *   {@code com.unboundid.ldap.sdk.unboundidds} package structure, are only
 *   supported for use against Ping Identity, UnboundID, and
 *   Nokia/Alcatel-Lucent 8661 server products.  These classes provide support
 *   for proprietary functionality or for external specifications that are not
 *   considered stable or mature enough to be guaranteed to work in an
 *   interoperable way with other types of LDAP servers.
 * </BLOCKQUOTE>
 */
@ThreadSafety(level=ThreadSafetyLevel.COMPLETELY_THREADSAFE)
public enum ErrorLogCategory
{
  /**
   * The error log category used for messages related to access control.
   */
  ACCESS_CONTROL,



  /**
   * The error log category used for messages related to the server
   * administration framework.
   */
  ADMIN,



  /**
   * The error log category used for messages related to tools used for
   * administering the server.
   */
  ADMIN_TOOL,



  /**
   * The error log category used for messages generated by most types of
   * Directory Server backends.
   */
  BACKEND,



  /**
   * The error log category used for messages related to the server
   * configuration.
   */
  CONFIG,



  /**
   * The error log category used for messages related to the core processing of
   * the server.
   */
  CORE,



  /**
   * The error log category used for messages related to the use of the dsconfig
   * tool.
   */
  DSCONFIG,



  /**
   * The error log category used for messages generated by server extensions.
   */
  EXTENSIONS,



  /**
   * The error log category used for messages generated by the backend using the
   * Berkeley DB Java Edition for storing data.
   */
  JEB,



  /**
   * The error log category used for messages generated by the logging
   * framework.
   */
  LOG,



  /**
   * The error log category used for messages generated by server plugins.
   */
  PLUGIN,



  /**
   * The error log category used for messages about communication performed with
   * clients.
   */
  PROTOCOL,



  /**
   * The error log category used for messages about the operation of the
   * Directory Proxy Server.
   */
  PROXY,



  /**
   * The error log category used for messages generated by the QuickSetup tool.
   */
  QUICKSETUP,



  /**
   * The error log category used for messages related to replication between
   * server instances.
   */
  REPLICATION,



  /**
   * The error log category used for messages related to information about the
   * environment in which the server is running.
   */
  RUNTIME_INFORMATION,



  /**
   * The error log category used for messages related to the server schema.
   */
  SCHEMA,



  /**
   * The error log category used for messages related to processing performed by
   * server tasks.
   */
  TASK,



  /**
   * The error log category used for messages generated by third-party
   * components.
   */
  THIRD_PARTY,



  /**
   * The error log category used for messages generated by server tools.
   */
  TOOLS,



  /**
   * The error log category used for messages generated by utility classes
   * within the server.
   */
  UTIL,



  /**
   * The error log category used for messages about the server version.
   */
  VERSION;



  /**
   * Retrieves the error log category with the specified name.
   *
   * @param  name  The name of the error log category to retrieve.  It must not
   *               be {@code null}.
   *
   * @return  The requested error log category, or {@code null} if no such
   *          category is defined.
   */
  @Nullable()
  public static ErrorLogCategory forName(@NotNull final String name)
  {
    switch (StaticUtils.toLowerCase(name))
    {
      case "accesscontrol":
      case "access-control":
      case "access_control":
        return ACCESS_CONTROL;
      case "admin":
        return ADMIN;
      case "admintool":
      case "admin-tool":
      case "admin_tool":
        return ADMIN_TOOL;
      case "backend":
        return BACKEND;
      case "config":
        return CONFIG;
      case "core":
        return CORE;
      case "dsconfig":
        return DSCONFIG;
      case "extensions":
        return EXTENSIONS;
      case "jeb":
        return JEB;
      case "log":
        return LOG;
      case "plugin":
        return PLUGIN;
      case "protocol":
        return PROTOCOL;
      case "proxy":
        return PROXY;
      case "quicksetup":
        return QUICKSETUP;
      case "replication":
        return REPLICATION;
      case "runtimeinformation":
      case "runtime-information":
      case "runtime_information":
        return RUNTIME_INFORMATION;
      case "schema":
        return SCHEMA;
      case "task":
        return TASK;
      case "thirdparty":
      case "third-party":
      case "third_party":
        return THIRD_PARTY;
      case "tools":
        return TOOLS;
      case "util":
        return UTIL;
      case "version":
        return VERSION;
      default:
        return null;
    }
  }
}
