package com.unity3d.ads.core.domain

import com.unity3d.ads.IUnityAdsTokenListener
import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.STATE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYNC
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZED
import com.unity3d.ads.core.data.model.InitializationState.NOT_INITIALIZED
import com.unity3d.ads.core.data.model.InitializationState.FAILED
import com.unity3d.ads.core.data.model.InitializationState.INITIALIZING
import com.unity3d.ads.core.data.repository.SessionRepository
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.AWAITED_INIT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.COMPLETE_STATE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_LISTENER_NULL
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.retrieveUnityCrashValue
import com.unity3d.services.core.misc.Utilities.wrapCustomerListener
import kotlinx.coroutines.TimeoutCancellationException
import kotlinx.coroutines.withTimeout
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
class CommonInitAwaitingGetHeaderBiddingToken(
    val getHeaderBiddingToken: GetHeaderBiddingToken,
    val sendDiagnosticEvent: SendDiagnosticEvent,
    val getInitializationState: GetInitializationState,
    val awaitInitialization: AwaitInitialization,
    val sessionRepository: SessionRepository
) : GetAsyncHeaderBiddingToken {
    val startTime = TimeSource.Monotonic.markNow()
    var listener: IUnityAdsTokenListener? = null
    private var didAwaitInit = false
    private var startState: InitializationState? = null

    override suspend fun invoke(listener: IUnityAdsTokenListener?) {
        this.listener = listener
        tokenStart()

        if (listener == null) {
            tokenFailure(reason = REASON_LISTENER_NULL, reasonDebug = "IUnityAdsTokenListener is null")
            return
        }

        if (!sessionRepository.shouldInitialize) {
            tokenFailure(reason = REASON_GATEWAY, reasonDebug = "!sessionRepository.shouldInitialize")
            return
        }

        val tokenTimeout = sessionRepository.nativeConfiguration.adOperations.getTokenTimeoutMs.toLong()

        try {
            withTimeout(tokenTimeout) {
                when (getInitializationState()) {
                    INITIALIZED, FAILED -> {
                        fetchToken()
                    }
                    NOT_INITIALIZED, INITIALIZING -> {
                        didAwaitInit = true
                        awaitInitialization()
                        if (!sessionRepository.shouldInitialize) {
                            tokenFailure(reason = REASON_GATEWAY, reasonDebug = "!sessionRepository.shouldInitialize")
                        } else {
                            fetchToken()
                        }
                    }
                }
            }
        } catch (e: TimeoutCancellationException) {
            fetchToken()
        }
    }

    private suspend fun fetchToken() {
        var reason: String? = null
        var reasonDebug: String? = null
        val token = try {
            getHeaderBiddingToken()
        } catch (e: Exception) {
            reason = SendDiagnosticEvent.REASON_UNCAUGHT_EXCEPTION
            reasonDebug = e.retrieveUnityCrashValue()
            null
        }
        if (token == null) {
            tokenFailure(reason, reasonDebug)
        } else {
            tokenSuccess(token)
        }
    }

    private fun tokenSuccess(token: String) {
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_SUCCESS,
            value = startTime.elapsedMillis(),
            tags = mapOf(
                SYNC to false.toString(),
                STATE to startState.toString(),
                COMPLETE_STATE to getInitializationState().toString(),
                AWAITED_INIT to didAwaitInit.toString(),
            )
        )
        wrapCustomerListener { listener?.onUnityAdsTokenReady(token) }
    }

    private fun tokenFailure(reason: String?, reasonDebug: String? = null) {
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_FAILURE,
            value = startTime.elapsedMillis(),
            tags = buildMap {
                put(SYNC, false.toString())
                put(STATE, startState.toString())
                put(COMPLETE_STATE, getInitializationState().toString())
                put(AWAITED_INIT, didAwaitInit.toString())
                reason?.let { put(REASON, reason) }
                reasonDebug?.let { put(REASON_DEBUG, reasonDebug) }
            })
        wrapCustomerListener { listener?.onUnityAdsTokenReady(null) }
    }

    private fun tokenStart() {
        startState = getInitializationState()
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_STARTED,
            tags = mapOf(
                SYNC to false.toString(),
                STATE to startState.toString()
            )
        )
    }
}