package com.unity3d.ads.core.data.datasource

import com.unity3d.ads.core.data.model.CacheError
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CacheSource
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.ads.core.domain.CreateFile
import com.unity3d.ads.core.domain.GetFileExtensionFromUrl
import com.unity3d.services.core.network.core.HttpClient
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.isSuccessful
import java.io.File

class AndroidRemoteCacheDataSource(
    private val createFile: CreateFile,
    private val getFileExtensionFromUrl: GetFileExtensionFromUrl,
    private val httpClient: HttpClient
) : CacheDataSource {
    override suspend fun getFile(
        cachePath: File,
        fileName: String,
        url: String?,
        priority: Int?
    ): CacheResult {
        if (url == null) {
            return CacheResult.Failure(CacheError.MALFORMED_URL, CacheSource.REMOTE)
        }
        val file = setupFile(cachePath, fileName)
        val response: HttpResponse = try {
            downloadFile(url, file, priority)
        } catch (e: Exception) {
            if (file.exists()) file.delete()
            return CacheResult.Failure(CacheError.NETWORK_ERROR, CacheSource.REMOTE)
        }
        val extension = if (url.isEmpty()) null else getFileExtensionFromUrl(url)
        val cachedFile = CachedFile(
            objectId = "",
            url = url,
            name = fileName,
            file = file,
            extension = extension ?: "",
            contentLength = response.contentSize,
            protocol = response.protocol,
            priority = priority ?: Int.MAX_VALUE
        )

        return if (response.isSuccessful()) {
            CacheResult.Success(cachedFile, CacheSource.REMOTE)
        } else {
            if (file.exists()) file.delete()
            CacheResult.Failure(CacheError.NETWORK_ERROR, CacheSource.REMOTE)
        }
    }

    private suspend fun downloadFile(url: String, destination: File, priority: Int?): HttpResponse {
        val request = HttpRequest(
            baseURL = url,
            downloadDestination = destination,
            priority = priority ?: Int.MAX_VALUE
        )
        return httpClient.execute(request)
    }

    private fun setupFile(cachePath: File, filename: String): File {
        val file = createFile(cachePath, filename)
        if (!file.exists()) {
            file.createNewFile()
        } else {
            file.delete()
            file.createNewFile()
        }
        return file
    }
}