package com.unity3d.services.core.network.core

import com.unity3d.ads.core.data.model.exception.NetworkTimeoutException
import com.unity3d.ads.core.data.model.exception.UnityAdsNetworkException
import com.unity3d.services.core.domain.ISDKDispatchers
import com.unity3d.services.core.network.model.HttpRequest
import com.unity3d.services.core.network.model.HttpResponse
import com.unity3d.services.core.network.model.RequestType
import kotlinx.coroutines.asExecutor
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.suspendCancellableCoroutine
import org.chromium.net.CronetEngine
import org.chromium.net.CronetException
import org.chromium.net.NetworkException
import org.chromium.net.UploadDataProviders
import org.chromium.net.UrlRequest
import org.chromium.net.UrlResponseInfo
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

class CronetClient(
    private val engine: CronetEngine,
    private val dispatchers: ISDKDispatchers,
): HttpClient {

    fun shutdown() {
        engine.shutdown()
    }

    override fun executeBlocking(request: HttpRequest): HttpResponse = runBlocking(dispatchers.io) {
        execute(request)
    }

    override suspend fun execute(request: HttpRequest): HttpResponse {
        return suspendCancellableCoroutine {
                cont ->
            val callback = object: UnityAdsUrlRequestCallback(dispatchers, request.readTimeout.toLong(), request.downloadDestination) {
                override fun onSucceeded(
                    request: UrlRequest,
                    info: UrlResponseInfo,
                    bodyBytes: ByteArray
                ) {
                    cont.resume(
                        HttpResponse(
                            statusCode = info.httpStatusCode,
                            headers = info.allHeaders,
                            urlString = info.url,
                            body = bodyBytes,
                            protocol = info.negotiatedProtocol,
                            client = NETWORK_CLIENT_CRONET,
                            contentSize = getContentSize(info)
                        )
                    )
                }

                override fun onFailed(
                    request: UrlRequest?,
                    info: UrlResponseInfo?,
                    error: CronetException?
                ) {
                    super.onFailed(request, info, error)

                    val cronetCode = (error as? NetworkException)?.cronetInternalErrorCode
                    val exception = UnityAdsNetworkException(
                        message = MSG_CONNECTION_FAILED,
                        code = info?.httpStatusCode,
                        url = info?.url,
                        protocol = info?.negotiatedProtocol,
                        cronetCode = cronetCode,
                        client = NETWORK_CLIENT_CRONET,
                    )
                    cont.resumeWithException(exception)
                }

                override fun onCanceled(request: UrlRequest?, info: UrlResponseInfo?) {
                    super.onCanceled(request, info)

                    val exception = NetworkTimeoutException(
                        message = MSG_CONNECTION_TIMEOUT,
                        url = info?.url,
                        protocol = info?.negotiatedProtocol,
                        client = NETWORK_CLIENT_CRONET,
                    )
                    cont.resumeWithException(exception)
                }
            }
            val cronetRequest = engine.newUrlRequestBuilder(buildUrl(request), callback, dispatchers.io.asExecutor())

            request.headers.forEach { (key, value) ->
                value.forEach {
                    cronetRequest.addHeader(key, it)
                }
            }
            if (request.method == RequestType.POST) {
                val body = when (request.body) {
                    is ByteArray -> request.body
                    is String -> request.body.toByteArray(Charsets.UTF_8)
                    else -> ByteArray(0)
                }
                cronetRequest.setUploadDataProvider(UploadDataProviders.create(body), dispatchers.io.asExecutor())
            }
            val req = cronetRequest
                .setHttpMethod(request.method.toString())
                .setPriority(getPriority(request.priority))
                .build()

            cont.invokeOnCancellation {
                req.cancel()
            }

            callback.startTimer(req)
            req.start()
        }
    }

    private fun getPriority(priority: Int): Int {
        return when (priority) {
            0 -> UrlRequest.Builder.REQUEST_PRIORITY_HIGHEST
            1 -> UrlRequest.Builder.REQUEST_PRIORITY_MEDIUM
            2 -> UrlRequest.Builder.REQUEST_PRIORITY_LOW
            else -> UrlRequest.Builder.REQUEST_PRIORITY_LOWEST
        }
    }

    private fun buildUrl(request: HttpRequest): String {
        return "${request.baseURL.trim('/')}/${request.path.trim('/')}".removeSuffix("/")
    }

    private fun getContentSize(info: UrlResponseInfo): Long {
        return info.allHeaders["Content-Length"]?.get(0)?.toLongOrNull() ?: -1L
    }

    companion object {
        private const val MSG_CONNECTION_TIMEOUT = "Network request timed out"
        private const val MSG_CONNECTION_FAILED = "Network request failed"
        private const val NETWORK_CLIENT_CRONET = "cronet"
    }
}