package com.unity3d.services

import android.app.Activity
import android.content.Context
import com.unity3d.ads.IUnityAdsLoadListener
import com.unity3d.ads.IUnityAdsTokenListener
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.UnityAdsShowOptions
import com.unity3d.ads.core.configuration.AlternativeFlowReader
import com.unity3d.ads.core.data.model.InitializationState
import com.unity3d.ads.core.data.model.Listeners
import com.unity3d.ads.core.domain.GetAdObject
import com.unity3d.ads.core.domain.GetAsyncHeaderBiddingToken
import com.unity3d.ads.core.domain.GetGameId
import com.unity3d.ads.core.domain.GetHeaderBiddingToken
import com.unity3d.ads.core.domain.GetInitializationState
import com.unity3d.ads.core.domain.InitializeBoldSDK
import com.unity3d.ads.core.domain.LegacyLoadUseCase
import com.unity3d.ads.core.domain.LegacyShowUseCase
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NOT_INITIALIZED
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_UNCAUGHT_EXCEPTION
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SOURCE_GET_TOKEN_API
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SOURCE_LOAD_API
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SOURCE_PUBLIC_API
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.STATE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.SYNC
import com.unity3d.ads.core.domain.SetInitializationState
import com.unity3d.ads.core.domain.ShouldAllowInitialization
import com.unity3d.ads.core.domain.om.OmFinishSession
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.getShortenedStackTrace
import com.unity3d.services.banners.UnityBannerSize
import com.unity3d.services.core.di.IServiceComponent
import com.unity3d.services.core.di.IServiceProvider
import com.unity3d.services.core.di.ServiceProvider
import com.unity3d.services.core.di.ServiceProvider.NAMED_GET_TOKEN_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_INIT_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_LOAD_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_OMID_SCOPE
import com.unity3d.services.core.di.ServiceProvider.NAMED_SHOW_SCOPE
import com.unity3d.services.core.di.get
import com.unity3d.services.core.di.inject
import com.unity3d.services.core.domain.task.EmptyParams
import com.unity3d.services.core.domain.task.InitializeSDK
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

/**
 * SDK Kotlin entry point
 */
@OptIn(ExperimentalTime::class)
class UnityAdsSDK(private val serviceProvider: IServiceProvider = ServiceProvider) : IServiceComponent {

    override fun getServiceProvider(): IServiceProvider {
        return serviceProvider
    }

    /**
     * Initialize the SDK
     */
    @Synchronized
    fun initialize(gameId: String?, source: String = SOURCE_PUBLIC_API): Job {
        val shouldAllowInitialization: ShouldAllowInitialization by inject()

        if (!shouldAllowInitialization(gameId)) return Job()

        val alternativeFlowReader: AlternativeFlowReader by inject()
        val initializeSDK: InitializeSDK by inject()
        val initializeBoldSDK: InitializeBoldSDK by inject()

        val initScope = get<CoroutineScope>(NAMED_INIT_SCOPE)

        return initScope.launch {
            val isAlternativeFlowEnabled = alternativeFlowReader()
            if (isAlternativeFlowEnabled) {
                initializeBoldSDK(source = source)
            } else {
                initializeSDK(EmptyParams)
            }

            initScope.cancel()
        }
    }

    fun load(
        placementId: String?,
        loadOptions: UnityAdsLoadOptions,
        listener: IUnityAdsLoadListener?,
        bannerSize: UnityBannerSize? = null
    ): Job {
        val loadScope = get<CoroutineScope>(NAMED_LOAD_SCOPE)
        val context: Context by inject()

        return loadScope.launch {
            val loadBoldSDK = get<LegacyLoadUseCase>()
            loadBoldSDK(context, placementId, loadOptions, listener, bannerSize)
            loadScope.cancel()
        }
    }

    fun show(activity: Activity, placementId: String?, showOptions: UnityAdsShowOptions?, listener: Listeners): Job {
        val showScope = get<CoroutineScope>(NAMED_SHOW_SCOPE)
        val showBoldSDK = get<LegacyShowUseCase>()

        return showScope.launch {
            showBoldSDK(
                activity = activity,
                placement = placementId,
                unityAdsShowOptions = showOptions,
                listeners = listener
            )
            showScope.cancel()
        }
    }

    fun getToken(): String? = runBlocking {
        fetchToken(true.toString())
    }

    fun getToken(listener: IUnityAdsTokenListener?): Job {
        val getAsyncHeaderBiddingToken: GetAsyncHeaderBiddingToken by inject()
        val getTokenScope = get<CoroutineScope>(NAMED_GET_TOKEN_SCOPE)

        return getTokenScope.launch {
            getAsyncHeaderBiddingToken(listener)
            getTokenScope.cancel()
        }
    }

    private suspend fun fetchToken(sync: String): String? {
        val getHeaderBiddingToken: GetHeaderBiddingToken by inject()
        val getInitializationState: GetInitializationState by inject()
        val sendDiagnosticEvent: SendDiagnosticEvent by inject()
        val startTime = TimeSource.Monotonic.markNow()
        sendDiagnosticEvent(
            event = SendDiagnosticEvent.HB_STARTED,
            tags = mapOf(
                SYNC to sync,
                STATE to getInitializationState().toString()
            ))
        var reason: String? = null
        var reasonDebug: String? = null
        val token = if (getInitializationState() != InitializationState.INITIALIZED){
            reason = REASON_NOT_INITIALIZED
            null
        } else {
            try {
                runBlocking { getHeaderBiddingToken() }
            } catch (e: Exception) {
                reason = REASON_UNCAUGHT_EXCEPTION
                reasonDebug = e.getShortenedStackTrace()
                null
            }
        }

        sendDiagnosticEvent(
            event = if (token == null) SendDiagnosticEvent.HB_FAILURE else SendDiagnosticEvent.HB_SUCCESS,
            value = startTime.elapsedMillis(),
            tags = buildMap {
                put(SYNC, sync)
                put(STATE, getInitializationState().toString())
                reason?.let{ put(REASON, reason) }
                reasonDebug?.let{ put(REASON_DEBUG, reasonDebug) }
            }
        )

        return token
    }

    fun finishOMIDSession(opportunityId: String): Job {
        val getAdObject: GetAdObject by inject()
        val omFinishSession: OmFinishSession by inject()
        val alternativeFlowReader: AlternativeFlowReader by inject()
        val omidScope = get<CoroutineScope>(NAMED_OMID_SCOPE)
        return omidScope.launch {
            val isAlternativeFlowEnabled = alternativeFlowReader()
            if (isAlternativeFlowEnabled) {
                val adObject = getAdObject(opportunityId)
                adObject?.let { omFinishSession(it) }
            }
            omidScope.cancel()
        }
    }
}