package com.unity3d.ads.core.data.repository

import android.content.Context
import com.unity3d.ads.core.data.datasource.CacheDataSource
import com.unity3d.ads.core.data.model.CacheError
import com.unity3d.ads.core.data.model.CacheResult
import com.unity3d.ads.core.data.model.CacheSource
import com.unity3d.ads.core.data.model.CachedFile
import com.unity3d.ads.core.domain.GetCacheDirectory
import com.unity3d.ads.core.domain.work.DownloadPriorityQueue
import com.unity3d.ads.core.extensions.getDirectorySize
import com.unity3d.ads.core.extensions.getSHA256Hash
import com.unity3d.services.UnityAdsConstants.DefaultUrls.CACHE_DIR_NAME
import com.unity3d.services.core.network.domain.CleanupDirectory
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.NonCancellable
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.filterNotNull
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.plus
import kotlinx.coroutines.withContext
import org.json.JSONArray
import java.io.File
import java.util.concurrent.ConcurrentHashMap

class AndroidCacheRepository(
    ioDispatcher: CoroutineDispatcher,
    private val getCacheDirectory: GetCacheDirectory,
    private val localCacheDataSource: CacheDataSource,
    private val remoteCacheDataSource: CacheDataSource,
    private val context: Context,
    private val sessionRepository: SessionRepository,
    private val cleanupDirectory: CleanupDirectory,
    private val downloadPriorityQueue: DownloadPriorityQueue,
) : CacheRepository {
    private val scope = CoroutineScope(ioDispatcher) + CoroutineName("CacheRepository") + NonCancellable
    // cachedFiles fileName to CachedFile
    val cachedFiles = ConcurrentHashMap<String, CachedFile>()
    // neededFiles fileName to ObjectID <String, Map<ObjectIds>>
    val neededFiles = ConcurrentHashMap<String, MutableSet<String>>()
    private val cacheDir: File = initCacheDir()

    override suspend fun getFile(
        url: String,
        objectId: String,
        headers: JSONArray?,
        priority: Int
    ): CacheResult = withContext(scope.coroutineContext) {
        val filename = getFilename(url)

        // Check if file is already in cache
        val localFile = localCacheDataSource.getFile(cacheDir, filename, url, priority)
        if (localFile is CacheResult.Success) {
            addFileToCache(localFile.cachedFile.copy(objectId = objectId))
            return@withContext localFile
        }

        // Download the file
        val fileResult = MutableStateFlow<CacheResult?>(null)
        downloadPriorityQueue(priority) {
            // Try local cache first
            val localFilePreemptive = localCacheDataSource.getFile(cacheDir, filename, url, priority)
            if (localFilePreemptive is CacheResult.Success) {
                addFileToCache(localFilePreemptive.cachedFile.copy(objectId = objectId))
                return@downloadPriorityQueue fileResult.update { localFilePreemptive }
            }

            // Try remote cache if local cache fails
            val remoteFile = remoteCacheDataSource.getFile(cacheDir, filename, url, priority)
            if (remoteFile is CacheResult.Success) {
                addFileToCache(remoteFile.cachedFile.copy(objectId = objectId))
            }
            fileResult.update { remoteFile }
        }

        return@withContext fileResult.filterNotNull().first()
    }

    override fun retrieveFile(fileName: String): CacheResult {
        val cachedFile = cachedFiles[fileName]
        return if (cachedFile != null) {
            CacheResult.Success(cachedFile, CacheSource.LOCAL)
        } else {
            CacheResult.Failure(CacheError.FILE_NOT_FOUND, CacheSource.LOCAL)
        }
    }

    override fun removeFile(cachedFile: CachedFile): Boolean {
        cachedFiles.remove(cachedFile.name)
        neededFiles[cachedFile.name]?.remove(cachedFile.objectId)
        return cachedFile.file?.takeIf(File::exists)?.delete() ?: false
    }

    override suspend fun doesFileExist(fileName: String): Boolean = cachedFiles.containsKey(fileName)

    fun getFilename(url: String): String = url.getSHA256Hash()

    override suspend fun clearCache() {
        withContext(scope.coroutineContext) {
            if (!sessionRepository.nativeConfiguration.hasCachedAssetsConfiguration()) {
                return@withContext cacheDir.listFiles()?.forEach(File::delete)
            }

            val config = sessionRepository.nativeConfiguration.cachedAssetsConfiguration
            cleanupDirectory(cacheDir, config.maxCachedAssetSizeMb, config.maxCachedAssetAgeMs)
        }
    }

    override suspend fun getCacheSize(): Long = withContext(scope.coroutineContext) {
        cacheDir.getDirectorySize()
    }

    private fun initCacheDir(): File {
        val dir = getCacheDirectory(getCacheDirBase(), getCacheDirPath())
        dir.mkdirs()
        return dir
    }

    private fun getCacheDirBase(): File = context.cacheDir

    private fun getCacheDirPath(): String = CACHE_DIR_NAME

    private fun addFileToCache(cachedFile: CachedFile) {
        cachedFiles[cachedFile.name] = cachedFile

        val observers = neededFiles[cachedFile.name] ?: mutableSetOf()
        observers.add(cachedFile.objectId)
        neededFiles[cachedFile.name] = observers
    }
}