package com.unity3d.ads.adplayer

import com.unity3d.ads.adplayer.model.WebViewEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_DEBUG
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.WEBVIEW_INVOCATION
import com.unity3d.ads.core.extensions.toTypedArray
import com.unity3d.services.core.log.DeviceLog
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import org.json.JSONArray
import org.json.JSONException

/**
 * A bridge between the webview and the native code
 *
 * @throws Exception if WebMessageListener is not supported
 */
class CommonWebViewBridge(
    dispatcher: CoroutineDispatcher,
    private val webViewContainer: WebViewContainer,
    adPlayerScope: CoroutineScope,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
) : WebViewBridge {
    val scope = adPlayerScope + dispatcher + CoroutineName("CommonWebViewBridge")
    private val callbacks = MutableStateFlow<Set<Pair<String, CompletableDeferred<Array<Any>>>>>(emptySet())

    private val _onInvocation = MutableSharedFlow<Invocation>(extraBufferCapacity = 64) // Default channel size
    override val onInvocation = _onInvocation.asSharedFlow()

    init {
        scope.launch {
            webViewContainer.addJavascriptInterface(this@CommonWebViewBridge, "webviewbridge")
        }
    }

    private suspend fun execute(handlerType: HandlerType, arguments: String) {
        webViewContainer.evaluateJavascript("window.nativebridge.${handlerType.jsPath}($arguments);")
    }

    override suspend fun sendEvent(event: WebViewEvent) {
        val arguments = JSONArray().apply {
            put(event.category)
            put(event.name)
            event.parameters.forEach { put(it) }
        }

        execute(HandlerType.EVENT, arguments.toString())
    }

    /**
     * Executes a method on the webview
     */
    override suspend fun request(className: String, method: String, vararg params: Any): Array<Any> {
        val callback = CompletableDeferred<Array<Any>>()
        val id = callback.hashCode().toString()
        callbacks.update { it + (id to callback) }

        val arguments = JSONArray().apply {
            put(className)
            put(method)
            put(id)
            params.forEach { put(it) }
        }

        execute(HandlerType.INVOCATION, arguments.toString())

        return callback.await()
    }

    /**
     * Handles a callback from the webview. This is called by the webview responding to a method call via [request]
     */
    override fun handleCallback(callbackId: String, callbackStatus: String, rawParameters: String) {
        val arrayParameters = JSONArray(rawParameters).toTypedArray()

        // todo: log error if callback is not found
        val entry = callbacks.value.find { (id, _) -> id == callbackId } ?: return
        val (_, callback) = entry

        if (callbackStatus in setOf("success", "error")) {
            sendDiagnosticEvent(SendDiagnosticEvent.OLD_CALLBACK_STATUS)
        }

        when (callbackStatus) {
            "OK", "success" -> callback.complete(arrayParameters)
            "ERROR", "error" -> callback.completeExceptionally(Exception(arrayParameters[0] as String))
        }

        callbacks.update { it - entry }
    }

    /**
     * Handles an invocation from the webview. This is called by the webview when it wants to call a method on the native side
     */
    override fun handleInvocation(message: String) {
        try {
            val invocationArray = try {
                JSONArray(message)
            } catch (ex: JSONException) {
                throw IllegalArgumentException("Invalid JSON array passed to CommonWebViewBridge: $message", ex)
            }

            for (idx in 0 until invocationArray.length()) {
                val currentInvocation = invocationArray[idx] as? JSONArray
                requireNotNull(currentInvocation) { "Invalid invocation passed to CommonWebViewBridge: $message" }
                require(currentInvocation.length() == 4) { "Invocation must have 4 elements: $currentInvocation" }

                val className = currentInvocation[0] as? String
                requireNotNull(className) { "Invalid class name passed to CommonWebViewBridge: $message" }

                val methodName = currentInvocation[1] as? String
                requireNotNull(methodName) { "Invalid method name passed to CommonWebViewBridge: $message" }

                val parameters = currentInvocation[2] as? JSONArray
                requireNotNull(parameters) { "Invalid parameters passed to CommonWebViewBridge: $message" }

                val callback = currentInvocation[3] as? String
                requireNotNull(callback) { "Invalid callback id passed to CommonWebViewBridge: $message" }

                val location = "$className.$methodName"

                DeviceLog.debug("Unity Ads WebView calling for: $location($parameters)")

                scope.launch {
                    val invocation = Invocation(location, parameters.toTypedArray())
                    _onInvocation.emit(invocation)
                    try {
                        withTimeout(5000) { invocation.isHandled.await() }
                        when (val result = invocation.getResult()) {
                            is WebViewEvent -> sendEvent(result)
                            else -> respond(callback, "OK", result)
                        }
                    } catch (e: Exception) {
                        val reason = when (e) {
                            is TimeoutCancellationException -> "Invocation($location) is not handled"
                            else -> e.message ?: e::class.java.simpleName
                        }
                        respond(callback, "ERROR", reason)
                    }
                }
            }
        } catch (t: Exception) {
            DeviceLog.error("Error handling invocation from webview ($message)")
            sendDiagnosticEvent(SendDiagnosticEvent.WEBVIEW_INVOCATION_ERROR, tags = mapOf(
                REASON_DEBUG to (t.message ?: t::class.java.simpleName),
                WEBVIEW_INVOCATION to message,
            ))
            throw IllegalArgumentException("Invalid message passed to CommonWebViewBridge: $message", t)
        }
    }

    /**
     * Responds to a callback from the webview
     */
    private suspend fun respond(callbackId: String, status: String, vararg params: Any) {
        val arguments = JSONArray().apply {
            put(callbackId)
            put(status)
            put(JSONArray(params))
        }

        execute(HandlerType.CALLBACK, "[$arguments]")
    }
}