package com.unity3d.ads.core.domain

import androidx.annotation.VisibleForTesting
import com.unity3d.ads.core.data.repository.FocusRepository
import com.unity3d.ads.core.data.repository.FocusState
import com.unity3d.ads.core.data.repository.SessionRepository
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import java.util.concurrent.ConcurrentHashMap
import kotlin.time.ComparableTimeMark
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource

@OptIn(ExperimentalTime::class)
class AndroidHandleFocusCounters(
    private val sessionRepository: SessionRepository,
    private val focusRepository: FocusRepository,
    private val isAdActivity: AndroidGetIsAdActivity,
    private val defaultDispatcher: CoroutineDispatcher,
    private val timeSource: TimeSource.WithComparableMarks = TimeSource.Monotonic,
) {
    @VisibleForTesting @Volatile var latestKnownActivityResumed: String? = null
    private val focusTimesPerActivity = ConcurrentHashMap<String, ComparableTimeMark>()
    private val previousFocusState = MutableStateFlow<FocusState?>(null)

    private fun onResume(activityName: String) {
        latestKnownActivityResumed = activityName
        focusTimesPerActivity[activityName] = timeSource.markNow()
    }

    private fun onPause(activityName: String) {
        val canUpdateCounter = latestKnownActivityResumed.let { it == null || it == activityName }

        if (canUpdateCounter) {
            val startMark = focusTimesPerActivity.remove(activityName) ?: timeSource.markNow()
            val elapsedTime = startMark.elapsedNow()
            sessionRepository.addTimeToGlobalAdsFocusTime(elapsedTime.inWholeMilliseconds.toInt())
        }
    }

    private fun onFocusStateChange(newState: FocusState) {
        val previousState = previousFocusState.getAndUpdate { newState } ?: return
        if (newState::class != previousState::class) {
            sessionRepository.incrementFocusChangeCount()
        }
    }

    operator fun invoke() {
        focusRepository.focusState.onEach { event ->
            onFocusStateChange(event)

            val activityName = event.activity.get()?.let { it::class.qualifiedName } ?: "unknown_activity_name"
            if (!isAdActivity(activityName)) return@onEach

            sessionRepository.incrementGlobalAdsFocusChangeCount()

            when (event) {
                is FocusState.Focused -> onResume(activityName)
                is FocusState.Unfocused -> onPause(activityName)
            }
        }.launchIn(CoroutineScope(defaultDispatcher))
    }
}