package com.unity3d.ads.core.domain

import android.content.Context
import android.util.Base64
import com.google.protobuf.ByteString
import com.unity3d.ads.UnityAds
import com.unity3d.ads.UnityAdsLoadOptions
import com.unity3d.ads.adplayer.AdPlayer
import com.unity3d.ads.adplayer.AdPlayerScope
import com.unity3d.ads.adplayer.model.LoadEvent
import com.unity3d.ads.core.data.model.AdObject
import com.unity3d.ads.core.data.model.LoadResult
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_COMMUNICATION_FAILURE
import com.unity3d.ads.core.data.model.LoadResult.Companion.MSG_NO_FILL
import com.unity3d.ads.core.data.repository.AdRepository
import com.unity3d.ads.core.data.repository.CampaignRepository
import com.unity3d.ads.core.data.repository.DeviceInfoRepository
import com.unity3d.ads.core.domain.LegacyLoadUseCase.Companion.KEY_AD_MARKUP
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_CREATE_AD_OBJECT_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_FILE_FAILURE_TIME
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_FILE_SUCCESS_TIME
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_WEBVIEW_FAILURE
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.LOAD_WEBVIEW_SUCCESS
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_AD_VIEWER
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_GATEWAY
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_INVALID_ENTRY_POINT
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NO_FILL
import com.unity3d.ads.core.domain.SendDiagnosticEvent.Companion.REASON_NO_WEBVIEW_ENTRY_POINT
import com.unity3d.ads.core.domain.events.GetOperativeEventApi
import com.unity3d.ads.core.extensions.elapsedMillis
import com.unity3d.ads.core.extensions.toBase64
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.UnityAdsConstants.Messages.MSG_INTERNAL_ERROR
import com.unity3d.services.core.properties.SdkProperties
import gatewayprotocol.v1.AdResponseOuterClass.AdResponse
import gatewayprotocol.v1.DiagnosticEventRequestOuterClass.DiagnosticAdType
import gatewayprotocol.v1.OperativeEventRequestOuterClass
import gatewayprotocol.v1.operativeEventErrorData
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.NonCancellable
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.single
import kotlinx.coroutines.withContext
import java.net.URI
import kotlin.time.DurationUnit
import kotlin.time.ExperimentalTime
import kotlin.time.TimeSource
import kotlin.time.measureTimedValue



@OptIn(ExperimentalTime::class)
internal class AndroidHandleGatewayAdResponse(
    private val adRepository: AdRepository,
    private val getWebViewContainerUseCase: AndroidGetWebViewContainerUseCase,
    private val getWebViewBridge: GetWebViewBridgeUseCase,
    private val deviceInfoRepository: DeviceInfoRepository,
    private val getHandleInvocationsFromAdViewer: HandleInvocationsFromAdViewer,
    private val campaignRepository: CampaignRepository,
    private val sendDiagnosticEvent: SendDiagnosticEvent,
    private val getOperativeEventApi: GetOperativeEventApi,
    private val getLatestWebViewConfiguration: GetLatestWebViewConfiguration,
    private val adPlayerScope: AdPlayerScope,
    private val getAdPlayer: GetAdPlayer,
    private val cacheWebViewAssets: CacheWebViewAssets,
) : HandleGatewayAdResponse {
    override suspend fun invoke(
        loadOptions: UnityAdsLoadOptions,
        opportunityId: ByteString,
        response: AdResponse,
        context: Context,
        placementId: String,
        adType: DiagnosticAdType,
        isHeaderBidding: Boolean
    ): LoadResult {
        var adPlayer: AdPlayer? = null

        try {
            if (response.hasError()) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE,
                    reason = REASON_GATEWAY,
                    reasonDebug = response.error.errorText
                )
            }

            if (response.adData.isEmpty) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.NO_FILL,
                    message = MSG_NO_FILL,
                    reason = REASON_NO_FILL
                )
            }

            val webviewConfiguration = getLatestWebViewConfiguration(
                receivedEntryPoint = response.webviewConfiguration.entryPoint,
                receivedVersion = response.webviewConfiguration.version,
                receivedAdditionalFiles = response.webviewConfiguration.additionalFilesList,
                receivedType = response.webviewConfiguration.type,
            )

            val tmpAdObject = AdObject(
                opportunityId = opportunityId,
                placementId = placementId,
                trackingToken = response.trackingToken,
                loadOptions = loadOptions,
                adType = adType,
                isHeaderBidding = isHeaderBidding
            )

            // Cache the webview assets (entry point and additional files)
            cacheWebViewAssets(webviewConfiguration)

            if (webviewConfiguration.entryPoint.isEmpty()) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE,
                    reason = REASON_NO_WEBVIEW_ENTRY_POINT
                )
            }

            val selectedUrl = SdkProperties.getConfigUrl().takeIf { it.endsWith(".html") }
                ?: webviewConfiguration.entryPoint

            val url = try {
                URI(selectedUrl)
            } catch (t: Throwable) {
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_COMMUNICATION_FAILURE,
                    reason = REASON_INVALID_ENTRY_POINT,
                    reasonDebug = selectedUrl
                )
            }
            val query =
                "${UnityAdsConstants.DefaultUrls.AD_PLAYER_QUERY_PARAMS}webviewType=${webviewConfiguration.type}&${url.query ?: ""}"
            val webViewUrl = selectedUrl.substringBeforeLast("?") + query
            val base64ImpressionConfiguration = Base64.encodeToString(
                response.impressionConfiguration.toByteArray(),
                Base64.NO_WRAP
            )
            val webviewContainerResult = measureTimedValue {
                runCatching { getWebViewContainerUseCase(adPlayerScope) }
            }.also { (result, duration) ->
                sendDiagnosticEvent(
                    if (result.isSuccess) LOAD_WEBVIEW_SUCCESS else LOAD_WEBVIEW_FAILURE,
                    duration.toDouble(DurationUnit.MILLISECONDS),
                    adObject = tmpAdObject
                )
            }
            val webviewContainer = webviewContainerResult.value.getOrThrow()
            val webviewBridge = getWebViewBridge(webviewContainer, adPlayerScope)
            adPlayer = getAdPlayer(webviewBridge, webviewContainer, opportunityId)

            val adObject = tmpAdObject.copy(
                adPlayer = adPlayer,
            )

            deviceInfoRepository.allowedPii
                .onEach { adPlayer.onAllowedPiiChange(it.toByteArray()) }
                .launchIn(adPlayer.scope)

            val loadAdViewerStartedTime = TimeSource.Monotonic.markNow()
            sendDiagnosticEvent(event = SendDiagnosticEvent.LOAD_STARTED_AD_VIEWER, adObject = adObject)

            getHandleInvocationsFromAdViewer(
                onInvocations = webviewBridge.onInvocation,
                adData = response.adData.toBase64(),
                impressionConfig = base64ImpressionConfiguration,
                adDataRefreshToken = response.adDataRefreshToken.toBase64(),
                adObject = adObject,
                onSubscription = {
                    measureTimedValue {
                        runCatching { webviewContainer.loadUrl(webViewUrl) }
                    }.also { (result, duration) ->
                        sendDiagnosticEvent(
                            if (result.isSuccess) LOAD_FILE_SUCCESS_TIME else LOAD_FILE_FAILURE_TIME,
                            duration.toDouble(DurationUnit.MILLISECONDS),
                            adObject = tmpAdObject
                        )
                        result.getOrThrow()
                    }
                },
            ).launchIn(adPlayer.scope)

            val loadEvent = adPlayer.onLoadEvent.single()

            if (loadEvent is LoadEvent.Error) {
                sendDiagnosticEvent(LOAD_CREATE_AD_OBJECT_FAILURE, loadAdViewerStartedTime.elapsedMillis(), adObject = adObject)
                withContext(NonCancellable) {
                    cleanup(Error(loadEvent.message), opportunityId, response, adPlayer)
                }
                return LoadResult.Failure(
                    error = UnityAds.UnityAdsLoadError.INTERNAL_ERROR,
                    message = MSG_INTERNAL_ERROR,
                    reason = REASON_AD_VIEWER,
                    reasonDebug = loadEvent.message,
                    isScarAd = adObject.isScarAd // This is the only time we can know if it was a SCAR ad that got parsed.
                )
            } else {
                sendDiagnosticEvent(
                    event = SendDiagnosticEvent.LOAD_CREATE_AD_OBJECT_SUCCESS,
                    value = loadAdViewerStartedTime.elapsedMillis(),
                    adObject = adObject
                )
            }

            campaignRepository.setLoadTimestamp(opportunityId)
            adRepository.addAd(opportunityId, adObject)
            if (loadOptions.objectId.isNullOrBlank() && loadOptions.data?.has(KEY_AD_MARKUP) == false) {
                adRepository.enqueueOpportunityForPlacement(placementId, opportunityId)
            }

            return LoadResult.Success(adObject)
        } catch (t: CancellationException) {
            withContext(NonCancellable) {
                cleanup(t, opportunityId, response, adPlayer)
            }
            throw t.cause ?: t
        }
    }

    private suspend fun cleanup(
        t: Throwable,
        opportunityId: ByteString,
        response: AdResponse,
        adPlayer: AdPlayer?
    ) {
        val operativeEventErrorData = operativeEventErrorData {
            errorType =
                OperativeEventRequestOuterClass.OperativeEventErrorType.OPERATIVE_EVENT_ERROR_TYPE_UNSPECIFIED
            message = t.cause?.message ?: t.message ?: ""
        }
        getOperativeEventApi(
            operativeEventType = OperativeEventRequestOuterClass.OperativeEventType.OPERATIVE_EVENT_TYPE_LOAD_ERROR,
            opportunityId = opportunityId,
            trackingToken = response.trackingToken,
            additionalEventData = operativeEventErrorData.toByteString()
        )
        adPlayer?.destroy()
    }
}
