package com.unity3d.ads.core.domain.billing

/**
 * Fetches product details with fallback strategy.
 * Tries the primary fetcher first, then falls back to the secondary fetcher if the primary fails.
 */
class ProductDetailsFetcherWithFallback(
    private val primaryFetcher: ProductDetailsFetcher,
    private val secondaryFetcher: ProductDetailsFetcher
) : ProductDetailsFetcher {
    
    override suspend fun fetchProductDetails(
        productId: String
    ): ProductDetailsResult {
        // Try primary fetcher first
        return when (val primaryResult = primaryFetcher.fetchProductDetails(productId)) {
            is ProductDetailsResult.Success -> primaryResult
            is ProductDetailsResult.NotFound -> {
                // If primary returns NotFound, try secondary
                secondaryFetcher.fetchProductDetails(productId)
            }
            is ProductDetailsResult.Failure -> {
                // If primary fails, try secondary
                val secondaryResult = secondaryFetcher.fetchProductDetails(productId)
                when (secondaryResult) {
                    is ProductDetailsResult.Success -> secondaryResult
                    else -> primaryResult // Return original failure if secondary also fails
                }
            }
        }
    }
}
